package main

import (
	"context"
	"fmt"
	"path"

	"github.com/jasonlvhit/gocron"
	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/cron"
	dynamicdicts "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
	"a.yandex-team.ru/travel/notifier/internal/processor"
	"a.yandex-team.ru/travel/notifier/internal/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/polling"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
	"a.yandex-team.ru/travel/notifier/internal/service/processing"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/queueobserver"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	rolloutservice "a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/scheduler"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts"
	ytlockwrapper "a.yandex-team.ru/travel/notifier/internal/ytlock"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	serviceName = "travel-notifier-processor"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &processor.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&processor.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		logger.Errorf("failed to create tvm client: %s", err)
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, processor.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgClient, err := processor.BuildPGClient(logger)
	if err != nil {
		logger.Errorf("failed to create PG client: %s", err)
		return
	}

	if err := models.MigrateAndInit(pgClient); err != nil {
		logger.Errorf(err.Error())
		return
	}

	clock := clockwork.NewRealClock()

	recipientsRepository := database.NewRecipientsRepository(pgClient)
	ordersRepository := database.NewOrdersRepository(pgClient)
	notificationsRepository := processor.BuildNotificationsRepository(pgClient)

	senderClient := processor.BuildSenderClient(logger)

	unifiedAgentClient, err := unifiedagent.NewGrpcClient(&processor.Cfg.UnifiedAgent, logger, nil)
	if err != nil {
		logger.Error("failed to create unified agent client", log.Error(err))
		return
	}

	ytDictsRegistry := ytdicts.NewRegistry(processor.Cfg.YtDicts, logger.WithName("YtDictsRegistry"))
	err = ytDictsRegistry.UpdateRegistryOnce(ctx)
	if err != nil {
		logger.Error("YT registry pre-start check failed", log.Error(err))
		if !processor.Cfg.YtDicts.IgnoreStartupErrors {
			return
		}
	}
	ytDictsRegistry.BackgroundRun()

	ordersClient := processor.BuildOrdersClient(logger)
	if ordersClient == nil {
		logger.Error("failed to create a client to the orders app")
		return
	}

	notificationsScheduler := scheduler.NewService(logger, notificationsRepository, recipientsRepository)

	dictsRegistry, err := dicts.NewRegistry(processor.Cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create dicts registry", log.Error(err))
		return
	}
	notificationsBuilder := scheduling.NewNotificationsBuilder(
		scheduling.NotificationsBuilderConfig{
			IsTesting:                 processor.Cfg.IsTesting(),
			TestingEmails:             containers.SetOf(processor.Cfg.Pretrip.Testing.Emails...),
			AdhocSendingInterval:      processor.Cfg.Pretrip.Testing.AdhocSendingInterval,
			WeekBeforeSendingInterval: processor.Cfg.Pretrip.Testing.WeekBeforeSendingInterval,
			DayBeforeSendingInterval:  processor.Cfg.Pretrip.Testing.DayBeforeSendingInterval,
		},
	)
	rollOutService := rolloutservice.NewService(processor.Cfg.RollOut)

	settlementDataProvider := extractors.NewSettlementDataProvider(
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetTimeZonesRepository(),
	)

	stationDataProvider := extractors.NewStationDataProvider(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
	)

	pretripProcessor, err := processor.BuildPretripProcessor(
		logger,
		notificationsRepository,
		recipientsRepository,
		ordersRepository,
		senderClient,
		ordersClient,
		tvmClient,
		dictsRegistry,
		notificationsBuilder,
		notificationsScheduler,
		clock,
		rollOutService,
		unifiedAgentClient,
		settlementDataProvider,
		stationDataProvider,
	)
	if err != nil {
		logger.Error("failed to create pretrip processor", log.Error(err))
		return
	}

	subscriptionsProcessor := subscriptions.NewProcessor(
		logger,
		ordersClient,
		notificationsRepository,
		recipientsRepository,
		senderClient,
		processor.Cfg.Subscriptions,
	)

	processingService := processing.NewService(
		logger,
		notificationsRepository,
		processing.WithProcessor(models.NotificationTypePretrip, pretripProcessor),
		processing.WithProcessor(models.NotificationTypePromoCode, subscriptionsProcessor),
		processing.WithConfig(processor.Cfg.Processing),
	)
	ytClient, err := ythttp.NewClient(
		&yt.Config{
			Proxy: processor.Cfg.Polling.YtLockCluster,
			Token: processor.Cfg.Polling.YtLockToken,
		},
	)
	pollingLockPath := ypath.Path(processor.Cfg.Polling.PollingLockPathPrefix + path.Join(processor.Cfg.EnvType, processor.Cfg.Polling.PollingLockDirectory))
	pollingLock := ytlock.NewLock(ytClient, pollingLockPath)
	notificationsFilter := polling.NewEmailFilter(logger, rollout.NewEmailFilter(processor.Cfg.Polling.EmailRules))
	notificationsPollingService := polling.NewNotificationsPollingService(
		logger,
		clock,
		notificationsRepository,
		notificationsFilter,
		processingService,
		processor.Cfg.Polling,
		ytlockwrapper.NewYtLock(pollingLock, ytlockwrapper.WithOnAcquire(polling.OnLockAcquired)),
	)
	cronScheduler := cron.NewScheduler(ctx, gocron.NewScheduler())
	if !processor.Cfg.Polling.Disabled {
		cronScheduler.Every(uint64(processor.Cfg.Polling.Interval.Seconds())).Seconds().Do(func() { notificationsPollingService.Poll(ctx) })
	}
	cronScheduler.Start()

	queueObserverLockPath := ypath.Path(
		processor.Cfg.QueueObserving.YtLockPathPrefix + path.Join(
			processor.Cfg.QueueObserving.Environment,
			processor.Cfg.QueueObserving.YtLockDirectory,
		),
	)
	queueObserverLock := ytlockwrapper.NewYtLock(ytlock.NewLock(ytClient, queueObserverLockPath))
	notificationsQueueObserver := queueobserver.NewNotificationsQueueObserverService(
		logger,
		processor.Cfg.QueueObserving,
		clock,
		notificationsRepository,
		queueObserverLock,
	)
	go notificationsQueueObserver.Observe(ctx)

	go func() {
		err = metricserver.RunMetricsHTTPServer(ctx, processor.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	waitShutdown := runShutdownServer(logger, ctxCancel, notificationsQueueObserver)
	runReadinessServer(pgClient, ytDictsRegistry)
	runProfilingService()
	runDynamicResourcesService(logger, dictsRegistry)

	waitShutdown()
	logger.Info("the instance has been shut down")
}

func runShutdownServer(
	logger *zap.Logger,
	ctxCancel context.CancelFunc,
	queueObserver *queueobserver.NotificationsQueueObserverService,
) (waitShutdown func()) {
	shutdownService := shutdown.NewService(
		processor.Cfg.Shutdown,
		func() {
			logger.Info("the instance is going to be shut down")
			queueObserver.Stop()
			ctxCancel()
		},
	)
	return shutdownService.BackroundRun()
}

func runReadinessServer(pgClient *pgclient.PGClient, ytDictsRegistry *ytdicts.Registry) {
	readinessService := readiness.NewService(
		processor.Cfg.Readiness,
		func() bool {
			_, err := pgClient.GetPrimary()
			if err != nil {
				return false
			}
			return ytDictsRegistry.AllReady()
		},
	)
	readinessService.BackroundRun()
}

func runProfilingService() {
	profilingService := profiling.NewService(processor.Cfg.Profiling)
	profilingService.BackroundRun()
}

func runDynamicResourcesService(logger *zap.Logger, dictsRegistry *dicts.Registry) {
	dynamicResourceService := dynamicdicts.NewService(
		logger,
		processor.Cfg.DynamicResources,
		dictsRegistry.OnUpdate,
	)
	dynamicResourceService.BackroundRun()
}
