package api

import (
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"
	"golang.yandex/hasql"
	gormlogger "gorm.io/gorm/logger"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/sender"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

func BuildPGClient(
	logger log.Logger,
) (*pgclient.PGClient, error) {
	logLevel := gormlogger.Silent
	if Cfg.IsDevelopment() {
		logLevel = gormlogger.Info
	}
	return pgclient.NewPGClient(
		Cfg.Database.Hosts,
		Cfg.Database.Port,
		Cfg.Database.Name,
		Cfg.Database.User,
		Cfg.Database.Password,
		pgclient.DefaultInitTimeout,
		[]pgclient.ClientOption{pgclient.WithOnCheckedNode(database.OnCheckedNode), pgclient.WithLogLevel(logLevel)},
		append(
			pgclient.DefaultClusterOptions, hasql.WithUpdateTimeout(Cfg.Database.HostsUpdateTimeout), hasql.WithTracer(
				hasql.Tracer{
					UpdatedNodes: database.OnUpdatedNodes,
					NodeDead:     database.OnDeadNode(logger),
				},
			),
		),
	)
}

func BuildNotificationsRepository(pgClient *pgclient.PGClient) *database.NotificationsRepository {
	gormDebugMode := Cfg.IsDevelopment()
	notificationsRepository := database.NewNotificationsRepository(pgClient, gormDebugMode)
	return notificationsRepository
}

func BuildBetterPriceSubscriptionRepository(pgClient *pgclient.PGClient) *database.BetterPriceSubscriptionsRepository {
	gormDebugMode := Cfg.IsDevelopment()
	betterPriceSubscriptionsRepository := database.NewBetterPriceSubscriptionsRepository(pgClient, gormDebugMode)
	return betterPriceSubscriptionsRepository
}

func BuildSenderClient(logger log.Logger) *sender.HTTPClient {
	return sender.NewHTTPClient(
		logger, sender.Config{
			HTTPClient: &http.Client{
				Timeout: Cfg.Sender.Timeout,
			},
			AuthKey:                Cfg.Sender.AuthKey,
			SenderURL:              Cfg.Sender.URL,
			Account:                Cfg.Sender.Account,
			AllowInactiveCampaigns: Cfg.Sender.AllowInactiveCampaigns,
		},
		sender.WithBackOffPolicy(
			func() backoff.BackOff {
				return &backoff.ExponentialBackOff{
					InitialInterval:     100 * time.Millisecond,
					RandomizationFactor: backoff.DefaultRandomizationFactor,
					Multiplier:          backoff.DefaultMultiplier,
					MaxInterval:         500 * time.Millisecond,
					MaxElapsedTime:      2 * Cfg.Sender.Timeout,
					Stop:                backoff.Stop,
					Clock:               backoff.SystemClock,
				}
			},
		),
		sender.WithRequestTimeout(Cfg.Sender.Timeout),
	)
}

func BuildOrdersClient(logger log.Logger) *orders.Client {
	return orders.NewClient(logger, Cfg.Tvm.SelfAppID, Cfg.OrdersApp, Cfg.GrpcClientConfig, orders.NewProtoToOrderInfoMapper())
}

func BuildOrderDestinationExtractor(dictsRegistry *dicts.Registry) *extractors.RoutePointsFromOrderExtractor {
	stationIDToSettlementIDMapper := extractors.NewStationIDToSettlementIDMapper(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
	)

	return extractors.NewRoutePointsFromOrderExtractor(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
		stationIDToSettlementIDMapper,
	)
}
