package api

import (
	"time"

	dynamicdicts "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/configtypes"
	"a.yandex-team.ru/travel/notifier/internal/contentadmin"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/mailsender/template"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/onlineregistration"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	"a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts"
)

type TvmConfig struct {
	Enabled   bool     `config:"TVM_ENABLED" yaml:"enabled"`
	WhiteList []uint32 `config:"TVM_WHITELIST" yaml:"whitelist"`
	SelfAppID uint32   `config:"TVM_SELFAPP_ID,required" yaml:"selfappid"`
}

type DebugConfig struct {
	Email               string `config:"DEBUG_EMAIL"`
	OrderID             string `config:"DEBUG_ORDER_ID"`
	NotificationSubtype string `config:"DEBUG_SUBTYPE"`
}

type Config struct {
	EnvType                string `config:"YENV_TYPE,required" yaml:"yenv_type"`
	TravelPortalURL        string `config:"TRAVEL_PORTAL_URL" yaml:"travel_portal_url"`
	Grpc                   grpcserver.GrpcConfig
	Metrics                metricsserver.MetricsConfig
	Logging                logging.Config
	Tvm                    TvmConfig
	HealthCheck            HealthCheckConfig
	Database               database.Config
	MailSender             template.MapperConfig
	OrdersApp              orders.OrdersAppConfig
	ContentAdmin           contentadmin.Config
	Pretrip                pretrip.PlanningConfig
	Shutdown               shutdown.Config
	Readiness              readiness.Config
	Sender                 SenderConfig
	Dicts                  dicts.Config
	YtDicts                ytdicts.Config `yaml:"yt_dicts"`
	Subscriptions          subscriptions.Config
	DynamicResources       dynamicdicts.Config `yaml:"dynamic_resources"`
	Debug                  DebugConfig
	RollOut                rollout.Config `yaml:"roll_out"`
	Profiling              profiling.Config
	UnifiedAgent           unifiedagent.ClientConfig    `yaml:"unified_agent"`
	GrpcClientConfig       configtypes.GrpcClientConfig `yaml:"grpc_client"`
	OnlineRegistration     onlineregistration.Config    `yaml:"online_registration"`
	HandleChangesFromQueue bool                         `config:"HANDLE_CHANGES_FROM_QUEUE" yaml:"handle_changes_from_queue"`
}

func (c Config) IsDevelopment() bool {
	return c.EnvType == "development"
}

func (c Config) IsTesting() bool {
	return c.EnvType == "testing"
}

type HealthCheckConfig struct {
	UpdateInterval time.Duration `config:"HEALTH_CHECK_UPDATE_INTERVAL" yaml:"update_interval"`
}

type SenderConfig struct {
	Timeout                time.Duration `config:"SENDER_TIMEOUT" yaml:"timeout"`
	AuthKey                string        `config:"SENDER_AUTHKEY" yaml:"authkey"`
	URL                    string        `config:"SENDER_URL" yaml:"url"`
	Account                string        `config:"SENDER_ACCOUNT" yaml:"account"`
	AllowInactiveCampaigns bool          `config:"SENDER_ALLOW_INACTIVE_CAMPAIGNS" yaml:"allow_inactive_campaigns"`
}

type RendererConfig struct {
	Timeout time.Duration `config:"RENDERER_TIMEOUT" yaml:"timeout"`
	URL     string        `config:"RENDERER_URL" yaml:"url"`
}

var (
	DefaultTvmConfig = TvmConfig{
		Enabled:   false,
		WhiteList: nil,
	}

	DefaultHealthCheckConfig = HealthCheckConfig{
		UpdateInterval: 1 * time.Second,
	}

	DefaultMetricsConfig = metricsserver.MetricsConfig{
		Addr:                       "[::]:8383",
		PerfMetricsRefreshInterval: 5 * time.Second,
		PerfMetricsPrefix:          "perf",
	}

	DefaultSenderConfig = SenderConfig{
		Timeout:                5 * time.Second,
		URL:                    "https://test.sender.yandex-team.ru",
		Account:                "travel",
		AllowInactiveCampaigns: false,
	}

	Cfg = Config{
		EnvType:                "development",
		Grpc:                   grpcserver.DefaultGrpcConfig,
		Metrics:                DefaultMetricsConfig,
		Logging:                logging.DefaultConfig,
		Tvm:                    DefaultTvmConfig,
		HealthCheck:            DefaultHealthCheckConfig,
		Database:               database.DefaultConfig,
		Pretrip:                pretrip.DefaultPlanningConfig,
		Shutdown:               shutdown.DefaultConfig,
		Readiness:              readiness.DefaultConfig,
		Sender:                 DefaultSenderConfig,
		Dicts:                  dicts.DefaultConfig,
		Subscriptions:          subscriptions.DefaultConfig,
		ContentAdmin:           contentadmin.DefaultConfig,
		TravelPortalURL:        "https://travel-test.yandex.ru",
		DynamicResources:       dynamicdicts.DefaultConfig,
		RollOut:                rollout.DefaultConfig,
		Profiling:              profiling.DefaultConfig,
		UnifiedAgent:           unifiedagent.DefaultClientConfig,
		OnlineRegistration:     onlineregistration.DefaultConfig,
		HandleChangesFromQueue: false,
	}
)
