package collector

import (
	"golang.yandex/hasql"
	gormlogger "gorm.io/gorm/logger"

	"a.yandex-team.ru/library/go/core/log"
	logbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/services/messagecollector"
	"a.yandex-team.ru/travel/notifier/internal/collector/dispatchers"
	"a.yandex-team.ru/travel/notifier/internal/collector/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/collector/unprocessed"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

func BuildCollector(
	logger log.Logger,
	cfg Config,
	logbrokerConfig LogbrokerConfig,
	orderChangesService *orderchanges.Service,
	unprocessedService *unprocessed.Service,
) *messagecollector.Collector {
	return messagecollector.NewCollector(logger).
		Route(
			cfg.OrdersCollectorEnabled,
			messagecollector.NewLogbrockerSource(
				logbroker.NewMultiEndpointConsumer(logger, cfg.OrdersCollector.Consumer, logbrokerConfig.Token),
			),
			dispatchers.NewOrder(
				logger,
				orderChangesService,
				unprocessedService,
			),
			messagecollector.WithLimiter(messagecollector.NewRPSLimiter(cfg.OrdersCollectorRPSLimit)),
			messagecollector.WithLimiter(messagecollector.NewInflightLimiter(cfg.OrdersCollectorInflightLimit)),
		).
		Route(
			cfg.UnprocessedOrdersCollectorEnabled,
			messagecollector.NewLogbrockerSource(
				logbroker.NewMultiEndpointConsumer(logger, cfg.UnprocessedOrdersCollector.Consumer, logbrokerConfig.Token),
			),
			dispatchers.NewUnprocessedOrders(
				logger,
				orderChangesService,
				unprocessedService,
			),
			messagecollector.WithLimiter(messagecollector.NewRPSLimiter(cfg.UnprocessedOrdersCollectorRPSLimit)),
			messagecollector.WithLimiter(messagecollector.NewInflightLimiter(cfg.UnprocessedOrdersCollectorInflightLimit)),
		)
}

func BuildPGClient(
	logger log.Logger,
) (*pgclient.PGClient, error) {
	logLevel := gormlogger.Silent
	if Cfg.IsDevelopment() {
		logLevel = gormlogger.Info
	}
	return pgclient.NewPGClient(
		Cfg.Database.Hosts,
		Cfg.Database.Port,
		Cfg.Database.Name,
		Cfg.Database.User,
		Cfg.Database.Password,
		pgclient.DefaultInitTimeout,
		[]pgclient.ClientOption{pgclient.WithOnCheckedNode(database.OnCheckedNode), pgclient.WithLogLevel(logLevel)},
		append(
			pgclient.DefaultClusterOptions, hasql.WithUpdateTimeout(Cfg.Database.HostsUpdateTimeout), hasql.WithTracer(
				hasql.Tracer{
					UpdatedNodes: database.OnUpdatedNodes,
					NodeDead:     database.OnDeadNode(logger),
				},
			),
		),
	)
}
