package collector

import (
	logbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/notifier/internal/collector/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/collector/unprocessed"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
)

type Config struct {
	EnvType   string `config:"YENV_TYPE,required" yaml:"yenv_type"`
	Logging   logging.Config
	Tvm       TvmConfig
	Metrics   metricsserver.MetricsConfig
	Logbroker LogbrokerConfig
	Shutdown  shutdown.Config
	Readiness readiness.Config
	Profiling profiling.Config

	OrderChangesService orderchanges.Config `yaml:"order_changes_service"`
	UnprocessedService  unprocessed.Config  `yaml:"unprocessed_service"`
	Database            database.Config

	OrdersCollectorEnabled       bool                 `config:"ORDERS_COLLECTOR_ENABLED" yaml:"orders_collector_enabled"`
	OrdersCollectorRPSLimit      float32              `config:"ORDERS_COLLECTOR_RPS_LIMIT" yaml:"orders_collector_rps_limit"`
	OrdersCollectorInflightLimit uint                 `config:"ORDERS_COLLECTOR_INFLIGHT_LIMIT" yaml:"orders_collector_inflight_limit"`
	OrdersCollector              QueueCollectorConfig `yaml:"orders_collector"`

	UnprocessedOrdersCollectorEnabled       bool                 `config:"UNPROCESSED_ORDERS_COLLECTOR_ENABLED" yaml:"unprocessed_orders_collector_enabled"`
	UnprocessedOrdersCollectorRPSLimit      float32              `config:"UNPROCESSED_ORDERS_COLLECTOR_RPS_LIMIT" yaml:"unprocessed_orders_collector_rps_limit"`
	UnprocessedOrdersCollectorInflightLimit uint                 `config:"UNPROCESSED_ORDERS_COLLECTOR_INFLIGHT_LIMIT" yaml:"unprocessed_orders_collector_inflight_limit"`
	UnprocessedOrdersCollector              QueueCollectorConfig `yaml:"unprocessed_orders_collector"`
}

func (c Config) IsDevelopment() bool {
	return c.EnvType == "development"
}

type TvmConfig struct {
	Enabled   bool     `config:"TVM_ENABLED" yaml:"enabled"`
	WhiteList []uint32 `config:"TVM_WHITELIST" yaml:"whitelist"`
	SelfAppID uint32   `config:"TVM_SELFAPP_ID,required" yaml:"selfappid"`
}

type QueueCollectorConfig struct {
	Consumer logbroker.ConsumerConfig `yaml:"consumer"`
}

type LogbrokerConfig struct {
	Token string `config:"LOGBROKER_TOKEN" yaml:"token"`
}

var (
	AllLogbrokerEndpoints = []string{
		"sas.logbroker.yandex.net",
		"vla.logbroker.yandex.net",
		"iva.logbroker.yandex.net",
	}

	DefaultLogbrokerConfig = LogbrokerConfig{}

	DefaultTvmConfig = TvmConfig{
		Enabled:   false,
		WhiteList: nil,
	}

	Cfg = Config{
		EnvType:   "development",
		Logging:   logging.DefaultConfig,
		Tvm:       DefaultTvmConfig,
		Logbroker: DefaultLogbrokerConfig,
		Shutdown:  shutdown.DefaultConfig,
		Readiness: readiness.DefaultConfig,
		Profiling: profiling.DefaultConfig,
		Metrics:   metricsserver.DefaultMetricsConfig,

		OrderChangesService: orderchanges.DefaultConfig,
		UnprocessedService:  unprocessed.DefaultConfig,
		Database:            database.DefaultConfig,

		OrdersCollectorEnabled:       false,
		OrdersCollectorRPSLimit:      0,
		OrdersCollectorInflightLimit: 0,
		OrdersCollector: QueueCollectorConfig{
			Consumer: logbroker.ConsumerConfig{
				Topic: logbroker.TopicConfig{
					Path:                 "/travel/test/orders-updates-queue",
					Consumer:             "/avia/development/trips/orders-consumer",
					MaxReadSize:          1024 * 1024,
					MaxReadMessagesCount: 100,
				},
				ReadEndpoints:         AllLogbrokerEndpoints,
				MinActiveConsumers:    2,
				Name:                  "Orders",
				DecompressionDisabled: true,
				Multiline:             false,
			},
		},

		UnprocessedOrdersCollectorEnabled:       false,
		UnprocessedOrdersCollectorRPSLimit:      0,
		UnprocessedOrdersCollectorInflightLimit: 0,
		UnprocessedOrdersCollector:              QueueCollectorConfig{},
	}
)
