package dispatchers

import (
	"context"
	"time"

	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/metrics"
	notifierorderapi "a.yandex-team.ru/travel/notifier/api/order/v1"
	"a.yandex-team.ru/travel/notifier/internal/collector/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/collector/unprocessed"
	"a.yandex-team.ru/travel/notifier/internal/collector/unprocessed/proto"
)

type base struct {
	logger              log.Logger
	orderChangesService *orderchanges.Service
	unprocessedService  *unprocessed.Service
}

func newBase(
	logger log.Logger,
	orderChangesService *orderchanges.Service,
	unprocessedService *unprocessed.Service,
) *base {
	return &base{
		logger:              logger,
		orderChangesService: orderChangesService,
		unprocessedService:  unprocessedService,
	}
}

func (b base) dispatchOrder(orderID string, collectedAt *timestamppb.Timestamp, retriesLeft uint32) {
	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	if orderID == "" {
		b.logger.Error("unable to dispatch empty orderID")
		return
	}

	if collectedAt == nil {
		collectedAt = timestamppb.Now()
	}

	err := b.orderChangesService.Handle(ctx, &notifierorderapi.OrderChangedReq{
		OrderId: orderID,
		Source:  notifierorderapi.OrderChangedSourceType_QUEUE,
	})
	if err == nil {
		b.logger.Info("order processed successfully", log.String("orderID", orderID))
		return
	}
	b.logger.Error(
		"unable to process order entity",
		log.Error(err),
		log.String("orderID", orderID),
	)

	unprocessedReq := &unprocessedproto.UnprocessedOrder{
		CollectedAt: collectedAt,
		RetriesLeft: retriesLeft - 1,
		OrderId:     orderID,
	}

	if retriesLeft > 0 {
		b.putAsUnprocessed(unprocessedReq)
	} else {
		b.storeFailedRequest(unprocessedReq)
	}
}

func (b base) putAsUnprocessed(req *unprocessedproto.UnprocessedOrder) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	if err := b.unprocessedService.Put(ctx, req); err != nil {
		b.logger.Error("unable to store unprocessed message", log.Error(err))
	}
}

func (b base) storeFailedRequest(req *unprocessedproto.UnprocessedOrder) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	err := b.unprocessedService.StoreFailed(ctx, req)
	if err != nil {
		b.logger.Error("failed to store unprocessed order to database", log.String("orderID", req.GetOrderId()))
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, failedToStore).Inc()
	} else {
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, stored).Inc()
	}
}
