package unprocessed

import (
	"context"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/jonboulle/clockwork"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/library/go/metrics"
	unprocessedproto "a.yandex-team.ru/travel/notifier/internal/collector/unprocessed/proto"
)

type LogbrokerProducer interface {
	Write(msg proto.Message) error
	Stat() persqueue.WriterStat
	Close() error
}

type Storage interface {
	Count(ctx context.Context) (int, error)
	Upsert(ctx context.Context, orderID string) (bool, error)
}

type Service struct {
	logger log.Logger

	logbrokerProducer LogbrokerProducer
	storage           Storage
	clock             clockwork.Clock
}

func NewService(logger log.Logger, logbrokerProducer LogbrokerProducer, clock clockwork.Clock, storage Storage) *Service {
	go sendProducerStat(logbrokerProducer)
	return &Service{logger: logger, logbrokerProducer: logbrokerProducer, clock: clock, storage: storage}
}

func (s *Service) Put(ctx context.Context, req *unprocessedproto.UnprocessedOrder) error {
	ctx = s.fillCtx(ctx, req)
	ctxlog.Info(ctx, s.logger, "putting unprocessed order")
	err := s.logbrokerProducer.Write(req)
	if err != nil {
		sendEventMetric(map[string]string{"status": "error", "phase": "write_to_logbroker", "order_type": orderTypeTravel})
		ctxlog.Error(ctx, s.logger, "failed to write proto-message into logbroker", log.Error(err))
		return err
	} else {
		sendEventMetric(map[string]string{"status": "success", "order_type": orderTypeTravel})
		ctxlog.Info(
			ctx,
			s.logger,
			"unprocessed order has been sent to logbroker successfully",
		)
	}
	return nil
}

func (s *Service) StoreFailed(ctx context.Context, message *unprocessedproto.UnprocessedOrder) error {
	orderID := message.GetOrderId()
	inserted, err := s.storage.Upsert(ctx, orderID)
	if err != nil {
		return nil
	}
	if inserted {
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, storedFailedOrders).Inc()
	}
	return nil
}

func (s *Service) MonitorFailedOrdersCount() {
	for range time.Tick(1 * time.Minute) {
		ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		count, err := s.storage.Count(ctx)
		cancel()
		if err != nil {
			s.logger.Error("failed to count failed unprocessed orders")
		} else {
			metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, totalFailed).Set(float64(count))
		}
	}
}

func (s *Service) PutOrderID(ctx context.Context, orderID string, retriesCount uint32) error {
	req := &unprocessedproto.UnprocessedOrder{
		CollectedAt: timestamppb.Now(),
		OrderId:     orderID,
		RetriesLeft: retriesCount,
	}
	return s.Put(ctx, req)
}

func (s *Service) fillCtx(ctx context.Context, req *unprocessedproto.UnprocessedOrder) context.Context {
	orderID := req.GetOrderId()
	retriesLeft := req.GetRetriesLeft()
	collectedAt := req.GetCollectedAt().AsTime()
	return ctxlog.WithFields(
		ctx,
		log.String("orderID", orderID),
		log.Time("collectedAt", collectedAt),
		log.UInt32("retriesLeft", retriesLeft),
	)
}

func sendProducerStat(producer LogbrokerProducer) {
	t := time.NewTicker(5 * time.Second)
	defer t.Stop()
	for range t.C {
		stat := producer.Stat()
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "inflight").Set(float64(stat.Inflight))
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "mem_usage").Set(float64(stat.MemUsage))
	}
}

func sendEventMetric(tags map[string]string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, tags, sentMessagesMetricName).Inc()
}
