package database

import (
	"context"
	"fmt"
	"time"

	"golang.org/x/sync/errgroup"
	"golang.yandex/hasql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

const (
	notificationsFilter        = "status = ? AND notify_at <= ? AND dispatch_type = ?"
	recipientFilter            = "recipient_id = ? AND notify_at >= ? AND notify_at <= ?  AND dispatch_type = ?"
	singleStatusFilter         = "status = ? AND dispatch_type = ?"
	singleStatusWithTypeFilter = "status = ? AND type_id = ? AND dispatch_type = ?"
	multiStatusFilter          = "status in (?) AND type_id = ? AND dispatch_type = ?"
	defaultOrder               = "notify_at asc, updated_at asc"
	// To prevent selecting an infinite number of notifications in the case our db has been corrupted
	maxNotificationsPerRecipientInTwoWeeks = 1000
)

var cancellableStatuses = []models.NotificationStatus{
	models.NotificationStatusPlanned,
	models.NotificationStatusPostponed,
	models.NotificationStatusReadyToSend,
}

type NotificationsRepository struct {
	pgClient *pgclient.PGClient
	debug    bool
}

func NewNotificationsRepository(pgClient *pgclient.PGClient, debug bool) *NotificationsRepository {
	return &NotificationsRepository{pgClient: pgClient, debug: debug}
}

func (r *NotificationsRepository) Create(ctx context.Context, notification models.Notification) (*models.Notification, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return r.createNotification(ctx, db, &notification)
		},
	)
	if err != nil {
		return nil, err
	}
	return &notification, nil
}

func (r *NotificationsRepository) CreateMany(ctx context.Context, notifications []models.Notification) ([]models.Notification, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			for _, n := range notifications {
				if err := r.createNotification(ctx, db, &n); err != nil {
					return err
				}
			}
			return nil
		},
	)
	if err != nil {
		return nil, err
	}
	return notifications, nil
}

func (r *NotificationsRepository) GetFirstN(ctx context.Context, now time.Time, plannedLimit, postponedLimit, notSentLimit uint) (
	[]models.Notification,
	error,
) {
	group := errgroup.Group{}
	planned := make([]models.Notification, 0, plannedLimit)
	postponed := make([]models.Notification, 0, postponedLimit)
	notSent := make([]models.Notification, 0, notSentLimit)
	group.Go(
		func() error {
			notifications, err := r.getFirstNPlanned(ctx, now, plannedLimit)
			if err != nil {
				return nil
			}
			planned = notifications
			return nil
		},
	)
	group.Go(
		func() error {
			notifications, err := r.getFirstNPostponed(ctx, now, postponedLimit)
			if err != nil {
				return nil
			}
			postponed = notifications
			return nil
		},
	)
	group.Go(
		func() error {
			notifications, err := r.getFirstNNotSent(ctx, notSentLimit)
			if err != nil {
				return nil
			}
			notSent = notifications
			return nil
		},
	)

	err := group.Wait()
	if err != nil {
		return nil, err
	}

	return append(append(planned, postponed...), notSent...), nil
}

func (r *NotificationsRepository) getFirstNPlanned(ctx context.Context, now time.Time, limit uint) ([]models.Notification, error) {
	return r.getFirstN(ctx, limit, notificationsFilter, models.NotificationStatusPlanned, now, models.DispatchTypePush)
}

func (r *NotificationsRepository) getFirstNPostponed(ctx context.Context, now time.Time, limit uint) ([]models.Notification, error) {
	return r.getFirstN(ctx, limit, notificationsFilter, models.NotificationStatusPostponed, now, models.DispatchTypePush)
}

func (r *NotificationsRepository) getFirstNNotSent(ctx context.Context, limit uint) ([]models.Notification, error) {
	return r.getFirstN(ctx, limit, singleStatusFilter, models.NotificationStatusReadyToSend, models.DispatchTypePush)
}

func (r *NotificationsRepository) getFirstN(ctx context.Context, limit uint, query interface{}, args ...interface{}) (
	[]models.Notification,
	error,
) {
	notifications := make([]models.Notification, 0, limit)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			if r.debug {
				db = db.Debug()
			}
			return db.
				WithContext(ctx).
				Preload(clause.Associations).
				Clauses(clause.Locking{Strength: "UPDATE", Options: "SKIP LOCKED"}).
				Model(&models.Notification{}).
				Where(query, args...).
				Limit(int(limit)).
				Order(defaultOrder).
				Find(&notifications).
				Error
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) CountAll(ctx context.Context) (int64, error) {
	return r.count(ctx, "")
}

func (r *NotificationsRepository) CountActualPlanned(ctx context.Context, now time.Time) (int64, error) {
	return r.count(ctx, notificationsFilter, models.NotificationStatusPlanned, now, models.DispatchTypePush)
}

func (r *NotificationsRepository) CountAllPlanned(ctx context.Context) (int64, error) {
	return r.count(ctx, singleStatusFilter, models.NotificationStatusPlanned, models.DispatchTypePush)
}

func (r *NotificationsRepository) CountPostponed(ctx context.Context, now time.Time) (int64, error) {
	return r.count(ctx, notificationsFilter, models.NotificationStatusPostponed, now, models.DispatchTypePush)
}

func (r *NotificationsRepository) CountReadyToSend(ctx context.Context) (int64, error) {
	return r.count(ctx, singleStatusFilter, models.NotificationStatusReadyToSend, models.DispatchTypePush)
}

func (r *NotificationsRepository) count(ctx context.Context, query interface{}, args ...interface{}) (int64, error) {
	var count int64
	err := r.pgClient.ExecuteInTransaction(
		hasql.Standby,
		func(db *gorm.DB) error {
			if r.debug {
				db = db.Debug()
			}
			return db.
				WithContext(ctx).
				Model(&models.Notification{}).
				Where(query, args...).
				Count(&count).
				Error
		},
	)
	return count, err
}

func (r *NotificationsRepository) CancelPlannedForOrder(ctx context.Context, orderID string, notificationType models.NotificationType, dispatchType models.DispatchType) (
	[]models.Notification,
	error,
) {
	notifications := make([]models.Notification, 0)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			if r.debug {
				db = db.Debug()
			}
			err := db.
				WithContext(ctx).
				Preload(clause.Associations).
				Clauses(clause.Locking{Strength: "UPDATE"}).
				Model(&models.Notification{}).
				Where(&models.Notification{OrderID: &orderID}).
				Where(multiStatusFilter, cancellableStatuses, notificationType.ID, dispatchType).
				Find(&notifications).
				Error
			if err != nil {
				return err
			}
			return r.changeStatusMany(ctx, db, notifications, models.NotificationStatusCancelled)
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) AlreadySentForOrder(ctx context.Context, orderID string, notificationType models.NotificationType) (
	[]models.Notification,
	error,
) {
	notifications := make([]models.Notification, 0)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			if r.debug {
				db = db.Debug()
			}
			return db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.Notification{}).
				Where(&models.Notification{OrderID: &orderID}).
				Where(singleStatusWithTypeFilter, models.NotificationStatusSent, notificationType.ID, models.DispatchTypePush).
				Find(&notifications).
				Error
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) GetForRecipient(
	ctx context.Context, recipientID int32, from time.Time, until time.Time,
) ([]models.Notification, error) {
	notifications := make([]models.Notification, 0)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			err := db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.Notification{}).
				Where(recipientFilter, recipientID, from, until, models.DispatchTypePush).
				Limit(maxNotificationsPerRecipientInTwoWeeks).
				Find(&notifications).
				Error
			if err != nil {
				return err
			}
			return nil
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) GetForOrder(
	ctx context.Context,
	orderID string,
	from time.Time,
	until time.Time,
	plannedOnly bool,
) ([]models.Notification, error) {
	notifications := make([]models.Notification, 0)

	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			query := db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.Notification{}).
				Joins("Order").
				Where("\"notifications\".\"notify_at\" <= ?", until).
				Where("\"notifications\".\"notify_at\" >= ?", from).
				Where("\"Order\".\"id\" = ?", orderID).
				Limit(maxNotificationsPerRecipientInTwoWeeks)
			if plannedOnly {
				query = query.Where(&models.Notification{Status: models.NotificationStatusPlanned})
			}
			err := query.Find(&notifications).Error
			if err != nil {
				return err
			}
			return nil
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) changeStatusMany(
	ctx context.Context,
	db *gorm.DB,
	notifications []models.Notification,
	targetStatus models.NotificationStatus,
) error {
	ids := make([]uint64, 0, len(notifications))
	for _, n := range notifications {
		ids = append(ids, n.ID)
	}
	return db.
		WithContext(ctx).
		Model(models.Notification{}).
		Where("id IN ?", ids).
		Updates(models.Notification{Status: targetStatus}).
		Error
}

func (r *NotificationsRepository) createNotification(ctx context.Context, db *gorm.DB, notification *models.Notification) error {
	db = db.WithContext(ctx)
	if err := db.FirstOrCreate(
		&notification.Type,
		models.NotificationType{Name: notification.Type.Name},
	).Error; err != nil {
		return err
	}
	if err := db.FirstOrCreate(
		&notification.Channel,
		models.NotificationChannel{Name: notification.Channel.Name},
	).Error; err != nil {
		return err
	}
	if notification.Recipient != nil {
		if err := db.FirstOrCreate(
			notification.Recipient,
			models.Recipient{ID: notification.Recipient.ID},
		).Error; err != nil {
			return err
		}
	}
	if notification.User != nil {
		if err := db.FirstOrCreate(
			notification.User,
			models.User{ID: notification.User.ID},
		).Error; err != nil {
			return err
		}
	}
	if notification.Order != nil {
		if err := db.FirstOrCreate(
			notification.Order,
			models.Order{ID: notification.Order.ID},
		).Error; err != nil {
			return err
		}
	}
	return db.Create(&notification).Error
}

func (r *NotificationsRepository) BeginTransaction(
	ctx context.Context,
	notification models.Notification,
	options TransactionOptions,
) (NotificationTransaction, error) {
	db, err := r.pgClient.GetPrimary()
	if err != nil {
		return nil, err
	}
	if r.debug {
		db = db.Debug()
	}

	db = db.Begin()
	if db.Exec(fmt.Sprintf("SET LOCAL statement_timeout = %d;", int(options.StatementTimeout.Milliseconds()))).Error != nil {
		return nil, db.Rollback().Error
	}
	if db.Exec(
		fmt.Sprintf(
			"SET LOCAL idle_in_transaction_session_timeout = %d;",
			int(options.IdleInTransactionSessionTimeout.Milliseconds()),
		),
	).Error != nil {
		return nil, db.Rollback().Error
	}

	lockingOptions := "NOWAIT"
	if options.WaitForLock {
		lockingOptions = ""
		if db.Exec(fmt.Sprintf("SET LOCAL lock_timeout = %d;", int(options.LockTimeout.Milliseconds()))).Error != nil {
			return nil, db.Rollback().Error
		}
	}

	db = db.
		WithContext(ctx).
		Model(&models.Notification{}).
		Where(&models.Notification{ID: notification.ID}).
		Clauses(clause.Locking{Strength: "UPDATE", Options: lockingOptions}).
		Preload(clause.Associations).
		Scan(&notification)
	// TODO(u-jeen): implement more viable rollback policy
	if db.Error != nil {
		db.Rollback()
		return nil, db.Error
	}
	metrics.GlobalAppMetrics().GetOrCreateGauge("transactions", nil, "in_progress").Add(1)
	return &notificationTransaction{tx: db, notification: &notification, isClosed: false}, db.Error
}

func (r *NotificationsRepository) GetPullNotificationsByPassportID(ctx context.Context, passportID string, now time.Time) ([]models.Notification, error) {
	notifications := make([]models.Notification, 0)

	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			query := db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.Notification{}).
				Joins("User").
				Where("\"notifications\".\"notify_at\" <= ?", now).
				Where("\"notifications\".\"deadline\" > ?", now).
				Where("\"User\".\"passport_id\" = ?", passportID).
				Where(&models.Notification{DispatchType: models.DispatchTypePull}).
				Where(&models.Notification{Status: models.NotificationStatusPlanned}).
				Limit(maxNotificationsPerRecipientInTwoWeeks)
			err := query.Find(&notifications).Error
			if err != nil {
				return err
			}
			return nil
		},
	)
	return notifications, err
}

func (r *NotificationsRepository) GetPullNotificationsByOrderIDs(ctx context.Context, orderIDs []string, now time.Time) ([]models.Notification, error) {
	notifications := make([]models.Notification, 0)
	if len(orderIDs) == 0 {
		return notifications, nil
	}
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			query := db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.Notification{}).
				Joins("User").
				Where("\"notifications\".\"notify_at\" <= ?", now).
				Where("\"notifications\".\"deadline\" > ?", now).
				Where("\"notifications\".\"order_id\" in (?)", orderIDs).
				Where(&models.Notification{DispatchType: models.DispatchTypePull}).
				Where(&models.Notification{Status: models.NotificationStatusPlanned}).
				Limit(maxNotificationsPerRecipientInTwoWeeks)
			err := query.Find(&notifications).Error
			if err != nil {
				return err
			}
			return nil
		},
	)
	return notifications, err
}
