package database

import (
	"context"
	"fmt"

	"golang.yandex/hasql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

type OrdersRepository struct {
	pgClient *pgclient.PGClient
}

func NewOrdersRepository(pgClient *pgclient.PGClient) *OrdersRepository {
	return &OrdersRepository{pgClient: pgClient}
}

func (r *OrdersRepository) GetByCorrelationID(ctx context.Context, correlationID string) ([]models.Order, error) {
	result := []models.Order{}
	if len(correlationID) == 0 {
		return result, fmt.Errorf("unable to get related orders by an empty correlation ID")
	}
	err := r.pgClient.ExecuteInTransaction(
		hasql.Alive,
		func(db *gorm.DB) error {
			return db.
				WithContext(ctx).
				Preload(clause.Associations).
				Where(&models.Order{CorrelationID: correlationID}).
				Find(&result).
				Error
		},
	)
	if err == gorm.ErrRecordNotFound {
		return []models.Order{}, nil
	}
	if err != nil {
		return result, err
	}
	return result, nil
}

func (r *OrdersRepository) Upsert(ctx context.Context, order models.Order) error {
	return r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.
				WithContext(ctx).
				Clauses(clause.OnConflict{UpdateAll: true}).
				Create(&order).
				Error
		},
	)
}

// Only used in tests
func (r *OrdersRepository) create(ctx context.Context, order models.Order) error {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).Create(&order).Error
		},
	)
	if err != nil {
		return err
	}
	return nil
}

func (r *OrdersRepository) GetByID(ctx context.Context, orderID string) (models.Order, error) {
	result := models.Order{}
	err := r.pgClient.ExecuteInTransaction(
		hasql.Alive, func(db *gorm.DB) error {
			return db.WithContext(ctx).
				Where(&models.Order{ID: orderID}).
				Find(&result).
				Error
		},
	)
	if err == nil || err == gorm.ErrRecordNotFound {
		return result, nil
	}
	return models.Order{}, err
}
