package database

import (
	"context"
	"fmt"

	"golang.yandex/hasql"
	"gorm.io/gorm"

	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

type RecipientsRepository struct {
	pgClient *pgclient.PGClient
}

func NewRecipientsRepository(pgClient *pgclient.PGClient) *RecipientsRepository {
	return &RecipientsRepository{pgClient: pgClient}
}

func (r *RecipientsRepository) Create(ctx context.Context, recipient models.Recipient) (*models.Recipient, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).Create(&recipient).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}

func (r *RecipientsRepository) GetOrCreateByEmail(ctx context.Context, email string) (*models.Recipient, error) {
	if len(email) == 0 {
		return nil, fmt.Errorf("unable to create recipient with an empty email")
	}
	recipient := models.NewRecipient().WithEmail(email)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).FirstOrCreate(
				&recipient,
				models.Recipient{Email: recipient.Email},
			).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}

func (r *RecipientsRepository) GetOrCreate(ctx context.Context, recipient models.Recipient) (*models.Recipient, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).FirstOrCreate(
				&recipient,
				models.Recipient{Email: recipient.Email},
			).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}

func (r *RecipientsRepository) Get(ctx context.Context, email string) (*models.Recipient, error) {
	if len(email) == 0 {
		return nil, fmt.Errorf("unable to get recipient by an empty email")
	}
	recipient := models.NewRecipient().WithEmail(email)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Alive,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).First(
				&recipient,
				models.Recipient{Email: recipient.Email},
			).Error
		},
	)
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}

func (r *RecipientsRepository) Update(ctx context.Context, recipient models.Recipient) (*models.Recipient, error) {
	if len(recipient.GetEmail()) == 0 {
		return nil, fmt.Errorf("unable to save recipient with an empty email")
	}
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).Save(&recipient).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}

func (r *RecipientsRepository) GetByHash(ctx context.Context, hash string) (*models.Recipient, error) {
	if len(hash) == 0 {
		return nil, fmt.Errorf("unable to get recipient by empty hash")
	}
	recipient := models.Recipient{}
	err := r.pgClient.ExecuteInTransaction(
		hasql.Alive,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).First(&recipient, models.Recipient{UnsubscribeHash: &hash}).Error
		},
	)
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}
	return &recipient, nil
}
