package database

import (
	"context"
	"time"

	"golang.yandex/hasql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

type UnprocessedOrdersStorage struct {
	pgClient *pgclient.PGClient
}

func NewUnprocessedOrdersStorage(pgClient *pgclient.PGClient) *UnprocessedOrdersStorage {
	return &UnprocessedOrdersStorage{pgClient: pgClient}
}

func (u *UnprocessedOrdersStorage) Count(ctx context.Context) (int, error) {
	var count int64 = 0
	err := u.pgClient.ExecuteInTransaction(hasql.PreferStandby, func(db *gorm.DB) error {
		return db.WithContext(ctx).Model(&models.UnprocessedOrder{}).Count(&count).Error
	})
	return int(count), err
}

func (u *UnprocessedOrdersStorage) Upsert(ctx context.Context, orderID string) (bool, error) {
	var existingOrder int64 = 0
	err := u.pgClient.ExecuteInTransaction(hasql.Primary, func(db *gorm.DB) error {
		model := &models.UnprocessedOrder{
			ID:        orderID,
			CreatedAt: time.Now().UTC(),
			UpdatedAt: time.Now().UTC(),
		}
		db = db.WithContext(ctx)

		if err := db.Model(&models.UnprocessedOrder{}).Where("id = ?", orderID).Count(&existingOrder).Error; err != nil {
			return err
		}
		return db.
			Clauses(clause.OnConflict{
				Columns:   []clause.Column{{Name: "id"}},
				DoUpdates: clause.AssignmentColumns([]string{"updated_at"})}).
			Create(&model).Error
	})
	return existingOrder == 0, err
}
