package database

import (
	"context"
	"fmt"
	"time"

	"github.com/cenkalti/backoff/v4"
	"golang.yandex/hasql"
	"gorm.io/gorm"

	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

type UsersRepository struct {
	pgClient *pgclient.PGClient
}

func NewUsersRepository(pgClient *pgclient.PGClient) *UsersRepository {
	return &UsersRepository{pgClient: pgClient}
}

func (u *UsersRepository) Create(ctx context.Context, user models.User) (*models.User, error) {
	err := u.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).Create(&user).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &user, nil
}

func (u *UsersRepository) GetOrCreate(ctx context.Context, user models.User) (*models.User, error) {
	getOrCreateBody := func() error {
		return u.pgClient.ExecuteInTransaction(
			hasql.Primary,
			func(db *gorm.DB) error {
				return db.WithContext(ctx).FirstOrCreate(
					&user,
					models.User{PassportID: user.PassportID},
				).Error
			},
		)
	}

	err := backoff.Retry(
		getOrCreateBody,
		backoff.WithContext(
			backoff.WithMaxRetries(
				backoff.NewConstantBackOff(time.Millisecond),
				2,
			),
			ctx,
		),
	)
	if err != nil {
		return nil, err
	}
	return &user, nil
}

func (u *UsersRepository) GetByPassportID(ctx context.Context, passportID string) (*models.User, error) {
	if len(passportID) == 0 {
		return nil, fmt.Errorf("unable to get user by an empty email")
	}
	user := models.NewUser().WithPassportID(passportID)
	err := u.pgClient.ExecuteInTransaction(
		hasql.Alive,
		func(db *gorm.DB) error {
			return db.WithContext(ctx).First(
				&user,
				models.User{PassportID: &passportID},
			).Error
		},
	)
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}
	return &user, nil
}
