package dicts

import (
	"fmt"
	"strings"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type SettlementsRepository struct {
	baseRepository     *repository.SettlementRepository
	settlementByGeoID  map[int]*rasp.TSettlement
	settlementByIATA   map[string]*rasp.TSettlement
	settlementBySirena map[string]*rasp.TSettlement
}

func NewSettlementsRepository() *SettlementsRepository {
	return &SettlementsRepository{
		baseRepository:     repository.NewSettlementRepository(),
		settlementByGeoID:  make(map[int]*rasp.TSettlement),
		settlementByIATA:   make(map[string]*rasp.TSettlement),
		settlementBySirena: make(map[string]*rasp.TSettlement),
	}
}

func (cr *SettlementsRepository) Write(b []byte) (int, error) {
	settlement := &rasp.TSettlement{}
	if err := proto.Unmarshal(b, settlement); err != nil {
		return 0, fmt.Errorf("SettlementsRepository:Write: %w", err)
	}
	cr.baseRepository.Add(settlement)
	if _, ok := cr.settlementByGeoID[int(settlement.GetGeoId())]; !ok {
		cr.settlementByGeoID[int(settlement.GetGeoId())] = settlement
	}
	if settlement.Iata != "" {
		cr.settlementByIATA[strings.ToUpper(settlement.Iata)] = settlement
	}
	if settlement.SirenaId != "" {
		cr.settlementByIATA[strings.ToUpper(settlement.SirenaId)] = settlement
	}
	return len(b), nil
}

func (cr *SettlementsRepository) Get(id int) (*rasp.TSettlement, bool) {
	settlement, ok := cr.baseRepository.Get(id)
	return settlement, ok
}

func (cr *SettlementsRepository) GetByGeoID(id int) (*rasp.TSettlement, bool) {
	result, found := cr.settlementByGeoID[id]
	return result, found
}

func (cr *SettlementsRepository) GetByCode(code string) (*rasp.TSettlement, bool) {
	if code == "" {
		return nil, false
	}
	code = strings.ToUpper(code)
	if result, found := cr.settlementByIATA[code]; found {
		return result, found
	}
	result, found := cr.settlementBySirena[code]
	return result, found
}

func (cr *SettlementsRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewSettlementsRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
