package dicts

import (
	"fmt"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type TimeZonesRepository struct {
	timeZoneByID map[int]*time.Location
	logger       log.Logger
}

func NewTimeZonesRepository(logger log.Logger) *TimeZonesRepository {
	return &TimeZonesRepository{
		timeZoneByID: make(map[int]*time.Location),
		logger:       logger,
	}
}

func (cr *TimeZonesRepository) Write(b []byte) (int, error) {
	timezone := &rasp.TTimeZone{}
	if err := proto.Unmarshal(b, timezone); err != nil {
		return 0, fmt.Errorf("TimeZonesRepository:Write: %w", err)
	}
	if _, ok := cr.timeZoneByID[int(timezone.GetId())]; !ok {
		if timezone.GetCode() != "" {
			tz, err := time.LoadLocation(timezone.GetCode())
			if err == nil {
				cr.timeZoneByID[int(timezone.GetId())] = tz
			} else {
				cr.logger.Warn(
					"Failed to load timezone",
					log.String("code", timezone.GetCode()),
					log.Int("id", int(timezone.GetId())),
				)
			}
		}
	}
	return len(b), nil
}

func (cr *TimeZonesRepository) Get(id int) (*time.Location, bool) {
	timeZone, ok := cr.timeZoneByID[id]
	return timeZone, ok
}

func (cr *TimeZonesRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newRepository := NewTimeZonesRepository(cr.logger)
	if err := iterator.Populate(newRepository); err != nil {
		return err
	}
	*cr = *newRepository
	return nil
}
