package externalhttp

import (
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/httputil/headers"
	tvmutil "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

type Config struct {
	ExternalAPIURL   string `yaml:"external_api_url"`
	ExternalAPITvmID uint32 `yaml:"external_api_tvm_id"`
}

type HTTPClient struct {
	backOffPolicyGetter func() backoff.BackOff
	httpClient          *http.Client
	tvmClient           tvm.Client
	config              Config
	requestTimeout      time.Duration
}

func NewHTTPClient(config Config, httpClient *http.Client, tvmClient tvm.Client, opts ...Option) *HTTPClient {
	c := &HTTPClient{
		config:         config,
		httpClient:     httpClient,
		tvmClient:      tvmClient,
		requestTimeout: 6 * time.Second,
	}
	defaultOpts := []Option{withDefaultBackOffPolicy, withDefaultRequestTimeout}
	for _, opt := range append(defaultOpts, opts...) {
		opt(c)
	}
	return c
}

func (c *HTTPClient) GetJSON(ctx context.Context, params map[string]string) ([]byte, error) {
	requestCtx, cancel := context.WithTimeout(ctx, c.requestTimeout)
	defer cancel()
	req, err := c.buildRequest(requestCtx)
	if err != nil {
		return nil, err
	}

	// add http params
	q := req.URL.Query()
	for k, v := range params {
		q.Add(k, v)
	}
	req.URL.RawQuery = q.Encode()

	var byteResponse []byte
	requestFunc := func() error {
		response, err := c.httpClient.Do(req)
		if err != nil {
			return RequestTravelAPIError{err}
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return RequestTravelAPIError{err}
		}
		if response.StatusCode == http.StatusNotFound {
			return backoff.Permanent(NewTravelAPINotFoundError(string(byteResponse)))
		}
		if response.StatusCode != http.StatusOK {
			err := fmt.Errorf("travel api answered with code %d: %s", response.StatusCode, string(byteResponse))
			if response.StatusCode == http.StatusBadRequest {
				return backoff.Permanent(err)
			}
			return RequestTravelAPIError{err}
		}
		return nil
	}
	if err = c.retryRequest(requestFunc); err != nil {
		return nil, RequestTravelAPIError{err}
	}
	return byteResponse, nil
}

func (c *HTTPClient) buildRequest(requestCtx context.Context) (*http.Request, error) {
	req, err := http.NewRequestWithContext(requestCtx, http.MethodGet, c.config.ExternalAPIURL, nil)
	if err != nil {
		return nil, err
	}
	if c.config.ExternalAPITvmID != 0 {
		tvmTicket, err := c.tvmClient.GetServiceTicketForID(requestCtx, tvm.ClientID(c.config.ExternalAPITvmID))
		if err != nil {
			return nil, err
		}
		req.Header.Set(tvmutil.XYaServiceTicket, tvmTicket)
	}
	req.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	return req, nil
}

func (c *HTTPClient) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}
