package extractors

import (
	"strconv"

	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/notifier/internal/orders"
)

const (
	// these constants were taken from
	// https://arcanum.yandex-team.ru/arc/trunk/arcadia/travel/api/src/main/java/ru/yandex/travel/api/services/hotels/geobase/GeoBaseHelpers.java?rev=r8099027#L29
	cityGeoIDType    = 6
	villageGeoIDType = 7
)

var geoIDSettlementTypes = containers.SetOf(
	cityGeoIDType,
	villageGeoIDType,
)

type RoutePointsFromOrderExtractor struct {
	stationsRepository      StationsRepository
	settlementsRepository   SettlementsRepository
	stationCodesRepository  StationCodesRepository
	settlementIDByStationID SettlementByStation
}

func NewRoutePointsFromOrderExtractor(
	stationsRepository StationsRepository,
	settlementsRepository SettlementsRepository,
	stationCodesRepository StationCodesRepository,
	settlementByStation SettlementByStation,
) *RoutePointsFromOrderExtractor {
	return &RoutePointsFromOrderExtractor{
		stationsRepository:      stationsRepository,
		settlementsRepository:   settlementsRepository,
		stationCodesRepository:  stationCodesRepository,
		settlementIDByStationID: settlementByStation,
	}
}

func (e *RoutePointsFromOrderExtractor) ExtractArrivalSettlementID(orderInfo *orders.OrderInfo) (id int, err error) {
	switch orderInfo.Type {
	case orders.OrderTypeAvia:
		return e.extractArrivalSettlementIDFromAvia(orderInfo.ID, orderInfo.AviaOrderItems)
	case orders.OrderTypeTrain:
		return e.extractArrivalSettlementIDFromTrain(orderInfo.ID, orderInfo.TrainOrderItems)
	case orders.OrderTypeHotel:
		return e.extractArrivalSettlementIDFromHotel(orderInfo.ID, orderInfo.HotelOrderItems)
	default:
		err = newErrUnknownOrderType(orderInfo.ID, orderInfo.Type.String())
		return
	}
}

func (e *RoutePointsFromOrderExtractor) ExtractDepartureSettlementID(orderInfo *orders.OrderInfo) (id int, err error) {
	switch orderInfo.Type {
	case orders.OrderTypeAvia:
		panic("not implemented")
	case orders.OrderTypeTrain:
		return e.extractDepartureSettlementIDFromTrain(orderInfo.ID, orderInfo.TrainOrderItems)
	case orders.OrderTypeHotel:
		panic("not implemented")
	default:
		err = newErrUnknownOrderType(orderInfo.ID, orderInfo.Type.String())
		return
	}
}

func (e *RoutePointsFromOrderExtractor) ExtractDepartureStationID(orderInfo *orders.OrderInfo) (id int, err error) {
	switch orderInfo.Type {
	case orders.OrderTypeAvia:
		return e.extractDepartureStationIDFromAvia(orderInfo.ID, orderInfo.AviaOrderItems)
	case orders.OrderTypeTrain:
		return e.extractDepartureStationIDFromTrain(orderInfo.ID, orderInfo.TrainOrderItems)
	case orders.OrderTypeHotel:
		panic("not implemented")
	default:
		err = newErrUnknownOrderType(orderInfo.ID, orderInfo.Type.String())
		return
	}
}

func (e *RoutePointsFromOrderExtractor) ExtractArrivalStationID(orderInfo *orders.OrderInfo) (id int, err error) {
	switch orderInfo.Type {
	case orders.OrderTypeAvia:
		panic("not implemented")
	case orders.OrderTypeTrain:
		return e.extractArrivalStationIDFromTrain(orderInfo.ID, orderInfo.TrainOrderItems)
	case orders.OrderTypeHotel:
		panic("not implemented")
	default:
		err = newErrUnknownOrderType(orderInfo.ID, orderInfo.Type.String())
		return
	}
}

func (e *RoutePointsFromOrderExtractor) extractArrivalSettlementIDFromAvia(orderID string, items []*orders.AviaOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	if len(item.OriginDestinations) == 0 || item.OriginDestinations[0] == nil {
		return 0, newErrNoDestination(orderID)
	}
	originDestination := item.OriginDestinations[0]
	if settlementID, err := e.getBySettlementCode(originDestination.ArrivalStation, orderID); err == nil {
		return settlementID, nil
	}
	if len(originDestination.Segments) == 0 {
		return 0, newErrNoDestination(orderID)
	}
	arrivalStation := originDestination.Segments[len(originDestination.Segments)-1].ArrivalStation
	if arrivalStation != "" && arrivalStation == originDestination.Segments[0].DepartureStation {
		return 0, newErrSameOriginDestination(orderID, arrivalStation)
	}
	return e.getByStationCode(arrivalStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) extractArrivalSettlementIDFromTrain(orderID string, items []*orders.TrainOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	return e.getByTrainStation(item.ArrivalStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) extractArrivalSettlementIDFromHotel(orderID string, items []*orders.HotelOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	geoRegions := item.GeoRegions
	if len(geoRegions) == 0 {
		return 0, newErrNoDestination(orderID)
	}
	geoID := 0
	for _, region := range geoRegions {
		if geoIDSettlementTypes.Contains(int(region.Type)) {
			geoID = int(region.GeoID)
			break
		}
	}
	if geoID == 0 {
		return 0, newErrHotelOrderWithoutSettlementGeoID(orderID)
	}
	settlement, found := e.settlementsRepository.GetByGeoID(geoID)
	if !found {
		return 0, newErrUnknownGeoID(orderID, geoID)
	}
	return int(settlement.Id), nil
}

func (e *RoutePointsFromOrderExtractor) extractDepartureSettlementIDFromTrain(orderID string, items []*orders.TrainOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	return e.getByTrainStation(item.DepartureStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) extractDepartureStationIDFromTrain(orderID string, items []*orders.TrainOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	return e.parseStationID(item.DepartureStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) extractDepartureStationIDFromAvia(orderID string, items []*orders.AviaOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	return e.getByStationCode(item.OriginDestinations[0].Segments[0].DepartureStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) extractArrivalStationIDFromTrain(orderID string, items []*orders.TrainOrderItem) (int, error) {
	if len(items) == 0 || items[0] == nil {
		return 0, newErrNoOrderItems(orderID)
	}
	item := items[0]
	return e.parseStationID(item.ArrivalStation, orderID)
}

func (e *RoutePointsFromOrderExtractor) getByTrainStation(rawStation string, orderID string) (int, error) {
	stationID, err := e.parseStationID(rawStation, orderID)
	if err != nil {
		return 0, err
	}
	return e.getByStationID(stationID, orderID)
}

func (e *RoutePointsFromOrderExtractor) getByStationCode(stationCode string, orderID string) (int, error) {
	if stationCode == "" {
		return 0, newErrInvalidDestination(orderID, stationCode)
	}
	stationID, found := e.stationCodesRepository.GetStationIDByCode(stationCode)
	if !found {
		return 0, newErrUnknownDestination(orderID, stationCode)
	}
	settlementID, found := e.settlementIDByStationID.Map(stationID)
	if !found {
		return 0, newErrStationWithoutSettlement(orderID, stationID)
	}
	return int(settlementID), nil
}

func (e *RoutePointsFromOrderExtractor) getBySettlementCode(settlementCode string, orderID string) (int, error) {
	if settlementCode == "" {
		return 0, newErrInvalidDestination(orderID, settlementCode)
	}
	settlement, found := e.settlementsRepository.GetByCode(settlementCode)
	if !found {
		return 0, newErrUnknownDestination(orderID, settlementCode)
	}
	return int(settlement.Id), nil
}

func (e *RoutePointsFromOrderExtractor) getByStationID(stationID int, orderID string) (int, error) {
	station, found := e.stationsRepository.Get(stationID)
	if !found {
		return 0, newErrUnknownDestination(orderID, stationID)
	}
	settlementID, found := e.settlementIDByStationID.Map(station.Id)
	if !found {
		return 0, newErrStationWithoutSettlement(orderID, station.Id)
	}
	return int(settlementID), nil
}

func (e *RoutePointsFromOrderExtractor) parseStationID(rawStation string, orderID string) (int, error) {
	if rawStation == "" {
		return 0, newErrInvalidDestination(orderID, rawStation)
	}
	if stationID, found := e.stationCodesRepository.GetStationIDByExpressCode(rawStation); found {
		return int(stationID), nil
	}
	stationID, err := strconv.Atoi(rawStation)
	if err != nil {
		return 0, newErrInvalidDestination(orderID, rawStation)
	}
	return stationID, nil
}
