package extractors

import (
	"fmt"
	"time"

	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/geo"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
)

type SettlementDataProvider struct {
	repository *dicts.SettlementsRepository
	timezones  *dicts.TimeZonesRepository
}

func (s *SettlementDataProvider) GetTitleTranslation(settlementID int) (*travel_commons_proto.TTranslationCaseRu, bool) {
	settlement, found := s.repository.Get(settlementID)
	if !found {
		return nil, false
	}
	return settlement.Title.Ru, true
}

func (s *SettlementDataProvider) GetSettlementTimeZone(settlementID int) *time.Location {
	settlement, found := s.repository.Get(settlementID)
	if !found {
		return nil
	}
	result, ok := s.timezones.Get(int(settlement.TimeZoneId))
	if !ok {
		return nil
	}
	return result
}

func NewSettlementDataProvider(
	repository *dicts.SettlementsRepository,
	timezones *dicts.TimeZonesRepository,
) *SettlementDataProvider {
	return &SettlementDataProvider{repository: repository, timezones: timezones}
}

func (s *SettlementDataProvider) GetNominativeTitle(id int) (title string, found bool) {
	if settlement, ok := s.repository.Get(id); ok {
		if settlement.Title != nil && settlement.Title.Ru != nil {
			title = settlement.Title.Ru.Nominative
		}
		if title == "" {
			title = settlement.TitleDefault
		}
	}
	return title, title != ""
}

func (s *SettlementDataProvider) GetAccusativeTitleWithPreposition(id int) (string, bool) {
	titleTranslation, ok := s.GetTitleTranslation(id)
	if !ok || titleTranslation.Accusative == "" {
		return "", false
	}
	return fmt.Sprintf("%s %s", s.GetPreposition(id), titleTranslation.Accusative), true
}

func (s *SettlementDataProvider) GetPreposition(id int) string {
	titleTranslation, ok := s.GetTitleTranslation(id)
	if !ok || titleTranslation.LocativePreposition == "" {
		return "в"
	}
	return titleTranslation.LocativePreposition
}

func (s *SettlementDataProvider) GetGeoID(id int) (geoID int, found bool) {
	settlement, found := s.repository.Get(id)
	if !found {
		return 0, found
	}
	return int(settlement.GeoId), found
}

func (s *SettlementDataProvider) GetCoordinates(id int) (latitude, longitude float64, found bool) {
	settlement, found := s.repository.Get(id)
	if !found {
		return 0, 0, found
	}
	return settlement.Latitude, settlement.Longitude, found
}

func (s *SettlementDataProvider) GetDistance(fromID, toID int) (float64, bool) {
	fromLatitude, fromLongitude, found := s.GetCoordinates(fromID)
	if !found {
		return 0, false
	}
	toLatitude, toLongitude, found := s.GetCoordinates(toID)
	if !found {
		return 0, false
	}
	km := geo.GetDistanceKM(fromLatitude, fromLongitude, toLatitude, toLongitude)
	return km, true
}
