package extractors

import (
	"strconv"

	"a.yandex-team.ru/travel/notifier/internal/geo"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type StationDataProvider struct {
	stationsRepository            StationsRepository
	stationToSettlementRepository StationToSettlementRepository
	settlementsRepository         SettlementsRepository
	stationCodesRepository        StationCodesRepository
}

func NewStationDataProvider(
	stationsRepository StationsRepository,
	stationToSettlementRepository StationToSettlementRepository,
	settlementsRepository SettlementsRepository,
	stationCodesRepository StationCodesRepository,
) *StationDataProvider {
	return &StationDataProvider{
		stationsRepository:            stationsRepository,
		stationToSettlementRepository: stationToSettlementRepository,
		settlementsRepository:         settlementsRepository,
		stationCodesRepository:        stationCodesRepository,
	}
}

func (s *StationDataProvider) GetStationByID(stationID int) (*rasp.TStation, bool) {
	station, found := s.stationsRepository.Get(stationID)
	if !found {
		return nil, false
	}
	return station, true
}

func (s *StationDataProvider) GetSettlementByStationID(stationID int) (*rasp.TSettlement, bool) {
	var settlementID int
	if stationToSettlement, ok := s.stationToSettlementRepository.Get(int(stationID)); ok && stationToSettlement.SettlementId > 0 {
		settlementID = int(stationToSettlement.SettlementId)
	} else {
		if station, ok := s.stationsRepository.Get(int(stationID)); ok && station.SettlementId > 0 {
			settlementID = int(station.SettlementId)
		}
	}
	if settlementID > 0 {
		settlement, ok := s.settlementsRepository.Get(settlementID)
		if ok {
			return settlement, true
		}
	}
	return nil, false
}

func (s *StationDataProvider) GetCoordinates(id int) (latitude, longitude float64, found bool) {
	settlement, found := s.stationsRepository.Get(id)
	if !found {
		return 0, 0, found
	}
	return settlement.Latitude, settlement.Longitude, found
}

func (s *StationDataProvider) GetDistance(fromID, toID int) (float64, bool) {
	fromLatitude, fromLongitude, found := s.GetCoordinates(fromID)
	if !found {
		return 0, false
	}
	toLatitude, toLongitude, found := s.GetCoordinates(toID)
	if !found {
		return 0, false
	}
	km := geo.GetDistanceKM(fromLatitude, fromLongitude, toLatitude, toLongitude)
	return km, true
}

func (s *StationDataProvider) getStationByRawCode(rawCode string) (*rasp.TStation, bool) {
	id, found := s.stationCodesRepository.GetStationIDByExpressCode(rawCode)
	if !found {
		if id, err := strconv.Atoi(rawCode); err == nil {
			return s.stationsRepository.Get(id)
		}
		return nil, false
	}
	return s.stationsRepository.Get(int(id))
}
