package handler

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/notifier/api/mailsender/v1"
	"a.yandex-team.ru/travel/notifier/internal/mailsender/template"
)

func NewGRPCMailSenderHandler(logger log.Logger, templateMapper templateMapper) *GRPCMailSenderHandler {
	return &GRPCMailSenderHandler{
		logger.WithName("GRPCMailSenderHandler"),
		templateMapper,
	}
}

type templateMapper interface {
	Map(templateCode string) template.TemplateSender
}

type GRPCMailSenderHandler struct {
	logger         log.Logger
	templateMapper templateMapper
}

func (h *GRPCMailSenderHandler) Send(ctx context.Context, req *pb.SendReq) (*pb.SendResp, error) {
	ctx, cancel := context.WithTimeout(ctx, time.Minute)
	defer cancel()

	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCMailSenderHandler:Send")
	defer span.Finish()

	if req.GetEmail() == "" {
		return nil, status.Error(codes.InvalidArgument, "empty email")
	}

	if req.GetMailTemplateCode() == "" {
		return nil, status.Error(codes.InvalidArgument, "empty template code")
	}

	sender := h.templateMapper.Map(req.GetMailTemplateCode())
	if sender == nil {
		return nil, status.Errorf(codes.InvalidArgument, "no template with code %s found", req.GetMailTemplateCode())
	}

	err := sender.Send(ctx, req.GetEmail(), req.GetMailParams())
	if err != nil {
		return nil, status.Error(codes.Unknown, err.Error())
	}
	return &pb.SendResp{}, nil
}

func (h *GRPCMailSenderHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterMailSenderAPIServer(server, h)
	}
}
