package template

import (
	"context"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/sender"
)

func (config *MapperConfig) NewTemplateSenderMapper(senderClient sender.Client, rendererClient *renderer.HTTPClient) (*templateMapper, error) {
	mapper := &templateMapper{
		make(map[string]TemplateSender, len(config.Templates)),
	}
	for templateCode, templateConfig := range config.Templates {
		templateSender, err := templateSenderFromConfig(templateConfig, senderClient, rendererClient)
		if err != nil {
			return nil, xerrors.Errorf("cannot create template mapper %s: %w", templateCode, err)
		}
		mapper.templateSenders[templateCode] = templateSender
	}

	return mapper, nil
}

type templateMapper struct {
	templateSenders map[string]TemplateSender
}

type unknownTemplateSenderCodeError struct {
	*errutil.TracedError
}

func templateSenderFromConfig(config Config, senderClient sender.Client, rendererClient *renderer.HTTPClient) (TemplateSender, error) {
	switch config.Type {
	case TSSender:
		return NewSenderTemplateSender(config.Params, senderClient)
	case TSRenderer:
		return NewRendererTemplateSender(config.Params, rendererClient)
	default:
		return nil, unknownTemplateSenderCodeError{errutil.NewTracedError(xerrors.Errorf(string(config.Type)))}
	}
}

func (tm *templateMapper) Map(templateCode string) TemplateSender {
	return tm.templateSenders[templateCode]
}

type TemplateSender interface {
	Send(ctx context.Context, email string, arguments map[string]string) error
}
