package models

import (
	"database/sql/driver"
	"fmt"
)

type DispatchType string

func (s DispatchType) String() string {
	return string(s)
}

const (
	DispatchTypePull DispatchType = "pull"
	DispatchTypePush DispatchType = "push"
)

var AllDispatchTypes = toDispatchTypeMap(
	[]DispatchType{DispatchTypePull, DispatchTypePush},
)

type DispatchTypeDBError struct {
	unexpectedValue string
}

func (e DispatchTypeDBError) Error() string {
	return fmt.Sprintf("unknown value of notification dispatch type: %v", e.unexpectedValue)
}

func (s *DispatchType) Scan(value interface{}) error {
	strValue := fmt.Sprintf("%v", value)
	if strValue == "" {
		*s = DispatchTypePush
		return nil
	}
	result, ok := AllDispatchTypes[strValue]
	if !ok {
		return DispatchTypeDBError{strValue}
	}
	*s = result
	return nil
}

func (s DispatchType) Value() (driver.Value, error) {
	return string(s), nil
}

func toDispatchTypeMap(DispatchTypes []DispatchType) map[string]DispatchType {
	result := make(map[string]DispatchType)
	for _, DispatchType := range DispatchTypes {
		result[string(DispatchType)] = DispatchType
	}
	return result
}
