package models

import (
	"time"

	"github.com/jackc/pgtype"
)

type NotificationStatus string

func (s NotificationStatus) String() string {
	return string(s)
}

const (
	NotificationStatusPlanned     NotificationStatus = "PLANNED"
	NotificationStatusPostponed   NotificationStatus = "POSTPONED"
	NotificationStatusReadyToSend NotificationStatus = "READY_TO_SEND"
	NotificationStatusSent        NotificationStatus = "SENT"
	NotificationStatusCancelled   NotificationStatus = "CANCELLED"
	NotificationStatusFailed      NotificationStatus = "FAILED"
)

var AllNotificationStatuses = []NotificationStatus{
	NotificationStatusPlanned,
	NotificationStatusPostponed,
	NotificationStatusReadyToSend,
	NotificationStatusSent,
	NotificationStatusCancelled,
	NotificationStatusFailed,
}

type Notification struct {
	ID           uint64 `gorm:"primaryKey;autoIncrement"`
	CreatedAt    time.Time
	UpdatedAt    time.Time          `gorm:"index:planned_index,priority:3;index:notifications_queue_index,priority:4"`
	NotifyAt     time.Time          `gorm:"index:planned_index,priority:2;index:notifications_queue_index,priority:3"`
	DispatchType DispatchType       `gorm:"type:varchar(20);not null;default:push;index:notifications_queue_index,priority:2"`
	Status       NotificationStatus `gorm:"type:varchar(100);index:planned_index,priority:1;index:notifications_queue_index,priority:1,option:CONCURRENTLY"`
	RecipientID  *int32
	Recipient    *Recipient `gorm:"foreignKey:RecipientID"`
	TypeID       int32
	Type         NotificationType    `gorm:"foreignKey:TypeID"`
	Subtype      NotificationSubtype `gorm:"type:varchar(100)"`
	ChannelID    int32
	Channel      NotificationChannel `gorm:"foreignKey:ChannelID"`
	OrderID      *string             `gorm:"index:order_id_index"`
	Order        *Order              `gorm:"foreignKey:OrderID"`
	Failures     uint
	Deadline     time.Time
	PromoCodeID  *uint
	PromoCode    *PromoCode   `gorm:"foreignKey:PromoCodeID"`
	Payload      *pgtype.JSON `gorm:"type:json"`
	UserID       *uint64      `gorm:"index:user_id_index"`
	User         *User        `gorm:"foreignKey:UserID"`
}

func NewNotification(
	notifyAt time.Time,
	deadline time.Time,
	status NotificationStatus,
	notificationType NotificationType,
	notificationChannel NotificationChannel,
	dispatchType DispatchType,
) Notification {
	return Notification{
		NotifyAt:     notifyAt,
		Deadline:     deadline,
		Status:       status,
		Type:         notificationType,
		Channel:      notificationChannel,
		DispatchType: dispatchType,
	}
}

func (n Notification) WithRecipient(recipient Recipient) Notification {
	localRecipient := recipient
	n.Recipient = &localRecipient
	return n
}

func (n Notification) WithUser(user User) Notification {
	localUser := user
	n.User = &localUser
	return n
}

func (n Notification) WithOrder(order Order) Notification {
	localOrder := order
	n.Order = &localOrder
	return n
}

func (n Notification) WithPayload(payload []byte) Notification {
	n.Payload = &pgtype.JSON{}
	_ = n.Payload.Set(payload)
	return n
}

func (n Notification) WithSubtype(subtype NotificationSubtype) Notification {
	n.Subtype = subtype
	return n
}

func (n Notification) WithPromoCode(promoCode PromoCode) Notification {
	n.PromoCode = &promoCode
	return n
}
