package models

import (
	"database/sql/driver"
	"fmt"
	"strings"
)

type NotificationSubtype string

func (s NotificationSubtype) String() string {
	return string(s)
}

const (
	NotificationWeekBefore  NotificationSubtype = "week-before"
	NotificationDayBefore   NotificationSubtype = "day-before"
	NotificationAdhoc       NotificationSubtype = "adhoc"
	NotificationPromoSend   NotificationSubtype = "promo-send"
	NotificationPromoRemind NotificationSubtype = "promo-remind"
)

var AllNotificationSubtypes = toNotificationSubtypeMap(
	[]NotificationSubtype{
		NotificationWeekBefore,
		NotificationDayBefore,
		NotificationAdhoc,
		NotificationPromoSend,
		NotificationPromoRemind,
	},
)

type NotificationSubtypeDBError struct {
	unexpectedValue string
}

func (e NotificationSubtypeDBError) Error() string {
	return fmt.Sprintf("unknown value of notification subtype: %v", e.unexpectedValue)
}

func (s *NotificationSubtype) Scan(value interface{}) error {
	strValue := fmt.Sprintf("%v", value)
	if strValue == "" {
		*s = NotificationAdhoc
		return nil
	}
	result, ok := AllNotificationSubtypes[strValue]
	if !ok {
		return NotificationSubtypeDBError{strValue}
	}
	*s = result
	return nil
}

func (s NotificationSubtype) Value() (driver.Value, error) {
	return string(s), nil
}

func toNotificationSubtypeMap(notificationSubtypes []NotificationSubtype) map[string]NotificationSubtype {
	result := make(map[string]NotificationSubtype)
	for _, notificationSubtype := range notificationSubtypes {
		result[string(notificationSubtype)] = notificationSubtype
	}
	return result
}

func (s *NotificationSubtype) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var o string
	err := unmarshal(&o)
	if err != nil {
		return err
	}

	if op, ok := AllNotificationSubtypes[strings.ToLower(o)]; !ok {
		return fmt.Errorf("could not convert %s to notification subtype", o)
	} else {
		*s = op
	}
	return nil
}
