package models

import (
	"database/sql/driver"
	"fmt"
	"strings"
)

type OrderType string

func (t OrderType) String() string {
	return string(t)
}

const (
	OrderAvia    OrderType = "avia"
	OrderTrain   OrderType = "train"
	OrderHotel   OrderType = "hotel"
	OrderUnknown OrderType = ""
)

var allOrderTypes = toOrderTypeMap([]OrderType{OrderUnknown, OrderAvia, OrderTrain, OrderHotel})

func (t *OrderType) Scan(value interface{}) error {
	strValue := fmt.Sprintf("%v", value)
	result, ok := allOrderTypes[strValue]
	if !ok {
		return OrderValueDBError{"OrderType", strValue}
	}
	*t = result
	return nil
}

func (t OrderType) Value() (driver.Value, error) {
	return t.String(), nil
}

func toOrderTypeMap(orderTypes []OrderType) map[string]OrderType {
	result := make(map[string]OrderType)
	for _, orderType := range orderTypes {
		result[orderType.String()] = orderType
	}
	return result
}

func (t *OrderType) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var o string
	err := unmarshal(&o)
	if err != nil {
		return err
	}

	if op, ok := allOrderTypes[strings.ToLower(o)]; !ok {
		return fmt.Errorf("could not convert %s to order type", o)
	} else {
		*t = op
	}

	return nil
}
