package models

import (
	"time"

	"a.yandex-team.ru/travel/hotels/proto/data_config/promo"
)

type PromoCodeForSubscription struct {
	Type         string
	Amount       float64
	MinTotalCost float64
}

type PromoConfig struct {
	PromoCode *PromoCodeForSubscription
}

type PromoEvent struct {
	EventID          string
	OrdersCampaignID string
	Enabled          bool
	StartAt          time.Time
	EndAt            time.Time
	Verticals        []string
	Sources          []string
}

func (e *PromoEvent) activeAt(now time.Time) bool {
	var afterStart, beforeEnd bool
	if e.StartAt.IsZero() {
		afterStart = true
	} else {
		afterStart = now.After(e.StartAt)
	}
	if e.EndAt.IsZero() {
		beforeEnd = true
	} else {
		beforeEnd = now.Before(e.EndAt)
	}
	return afterStart && beforeEnd
}

func (e *PromoEvent) verticalEnabled(vertical string) bool {
	for _, v := range e.Verticals {
		if v == vertical {
			return true
		}
	}
	return false
}

func (e *PromoEvent) Active(now time.Time, vertical string) bool {
	return e.Enabled && e.activeAt(now) && e.verticalEnabled(vertical)
}

func PromoEventFromProto(event *promo.TPromoEvent) PromoEvent {
	return PromoEvent{
		EventID:          event.GetEventId(),
		OrdersCampaignID: event.GetOrdersCampaignId(),
		Enabled:          event.GetEnabled(),
		StartAt:          parseRFC3339TimeOrZero(event.GetStartAt()),
		EndAt:            parseRFC3339TimeOrZero(event.GetEndAt()),
		Verticals:        event.Verticals,
		Sources:          event.Sources,
	}
}

func parseRFC3339TimeOrZero(datetime string) time.Time {
	parse, err := time.Parse(time.RFC3339, datetime)
	if err != nil {
		return time.Time{}
	}
	return parse
}
