package models

import (
	"time"

	"a.yandex-team.ru/library/go/ptr"
)

type Recipient struct {
	ID                   int32 `gorm:"primaryKey;autoIncrement"`
	CreatedAt            time.Time
	UpdatedAt            time.Time
	Email                *string `gorm:"unique"`
	IsSubscribed         bool
	SubscriptionSource   *string
	SubscriptionVertical *string
	SubscribedAt         *time.Time
	UnsubscribeHash      *string `gorm:"unique"`
	UnsubscribedAt       *time.Time
	NationalVersion      *string
	Language             *string
	Timezone             *string
}

func NewRecipient() Recipient {
	return Recipient{}
}

func (r Recipient) WithEmail(email string) Recipient {
	r.Email = &email
	return r
}

func (r Recipient) GetEmail() string {
	if r.Email != nil {
		return *r.Email
	}
	return ""
}

func (r *Recipient) Subscribe(
	source, travelVertical, nationalVersion, language, timezone string,
	subscribedAt time.Time,
	unsubscribeHashGenerator func(string) string,
) {
	if r.Email == nil {
		return
	}
	r.SubscriptionSource = &source
	r.SubscriptionVertical = &travelVertical
	r.NationalVersion = &nationalVersion
	r.Language = &language
	r.Timezone = &timezone
	r.SubscribedAt = &subscribedAt
	r.IsSubscribed = true
	if r.UnsubscribeHash == nil {
		r.UnsubscribeHash = ptr.String(unsubscribeHashGenerator(*r.Email))
	}
}

func (r *Recipient) Unsubscribe(unsubscribedAt time.Time) {
	r.UnsubscribedAt = &unsubscribedAt
	r.IsSubscribed = false
}

func (r Recipient) GetSubscriptionStatus() SubscirptionStatus {
	formatTime := func(t *time.Time) string {
		if t == nil {
			return ""
		}
		return t.Format(time.RFC3339)
	}
	status := SubscirptionStatus{
		IsSubscribed:   r.IsSubscribed,
		SubscribedAt:   formatTime(r.SubscribedAt),
		UnsubscribedAt: formatTime(r.UnsubscribedAt),
	}
	if r.SubscriptionSource != nil {
		status.Source = *r.SubscriptionSource
	}
	if r.SubscriptionVertical != nil {
		status.Vertical = *r.SubscriptionVertical
	}
	return status
}
