package models

import (
	"fmt"

	"a.yandex-team.ru/travel/notifier/internal/pgclient"
)

func MigrateAndInit(pgClient *pgclient.PGClient) error {
	if err := autoMigrateAll(pgClient); err != nil {
		return err
	}
	return initDefaultState(pgClient)
}

func DropAll(pgClient *pgclient.PGClient) error {
	rwDB, err := pgClient.GetPrimary()
	if err != nil {
		return errDrop{err}
	}
	err = rwDB.Migrator().DropTable(
		&NotificationChannel{},
		&NotificationType{},
		&PromoCode{},
		&Notification{},
		&BetterPriceSubscription{},
		&Variant{},
		&Recipient{},
		&User{},
		&UnprocessedOrder{},
	)
	if err != nil {
		return errDrop{err}

	}
	return nil
}

func autoMigrateAll(pgClient *pgclient.PGClient) error {
	rwDB, err := pgClient.GetPrimary()
	if err != nil {
		return errMigrateModels{err}
	}
	if err := rwDB.AutoMigrate(
		&NotificationChannel{},
		&NotificationType{},
		&Order{},
		&PromoCode{},
		&Notification{},
		&Recipient{},
		&User{},
		&Variant{},
		&BetterPriceSubscription{},
		&UnprocessedOrder{},
	); err != nil {
		return errMigrateModels{err}
	}
	return nil
}

func initDefaultState(pgClient *pgclient.PGClient) error {
	rwDB, err := pgClient.GetPrimary()
	if err != nil {
		return errInitialize{err}
	}
	rwDB = rwDB.Debug()
	for _, notificationChannel := range notificationChannels {
		if err := rwDB.FirstOrCreate(notificationChannel, NotificationChannel{Name: notificationChannel.Name}).Error; err != nil {
			return errInitialize{err}
		}
	}
	for _, notificationType := range notificationTypes {
		if err := rwDB.FirstOrCreate(notificationType, NotificationType{Name: notificationType.Name}).Error; err != nil {
			return errInitialize{err}
		}
	}
	return nil
}

type errMigrateModels struct {
	err error
}

func (e errMigrateModels) Error() string {
	return fmt.Sprintf("failed to automigrate database models: %s", e.err)
}

type errInitialize struct {
	err error
}

func (e errInitialize) Error() string {
	return fmt.Sprintf("failed to initialize default entities: %s", e.err)
}

type errDrop struct {
	err error
}

func (e errDrop) Error() string {
	return fmt.Sprintf("failed to drop db: %s", e.err)
}
