package handler

import (
	"context"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/notifier/api/order/v1"
	"a.yandex-team.ru/travel/notifier/internal/orderchanges"
)

type GRPCOrderChangesHandler struct {
	logger         log.Logger
	onOrderChanged orderchanges.OnOrderChanged
}

func NewGRPCOrderChangesHandler(l log.Logger, onOrderChanged orderchanges.OnOrderChanged) *GRPCOrderChangesHandler {
	return &GRPCOrderChangesHandler{logger: l, onOrderChanged: onOrderChanged}
}

func (h *GRPCOrderChangesHandler) OrderChanged(ctx context.Context, request *pb.OrderChangedReq) (
	response *pb.OrderChangedRsp,
	err error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCOrderChangesHandler:OrderChanged")
	defer span.Finish()

	h.logger.Info("OrderChanged request", log.String("orderID", request.OrderId))
	defer func() {
		h.logger.Info(
			"OrderChanged response",
			log.String("orderID", request.OrderId),
			log.Error(err),
		)
	}()

	req := orderchanges.OrderChangedRequest{
		OrderID: request.OrderId,
		Source:  h.mapOrderChangedSource(request.Source),
	}
	onOrderChangedResult := h.onOrderChanged(ctx, req)

	response = &pb.OrderChangedRsp{}
	switch onOrderChangedResult.Status {
	case orderchanges.OrderChangedStatusOK:
		return response, nil
	case orderchanges.OrderChangedStatusTemporaryError:
		return nil, status.Error(codes.Unavailable, onOrderChangedResult.Message)
	case orderchanges.OrderChangedStatusFailure:
		return nil, status.Error(codes.FailedPrecondition, onOrderChangedResult.Message)
	}
	return response, nil
}

func (h *GRPCOrderChangesHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterOrderChangesServiceServer(server, h)
	}
}

func (h *GRPCOrderChangesHandler) mapOrderChangedSource(source pb.OrderChangedSourceType) orderchanges.OrderChangedSource {
	switch source {
	case pb.OrderChangedSourceType_QUEUE:
		return orderchanges.OrderChangedSourceQueue
	default:
		return orderchanges.OrderChangedSourceSync
	}
}
