package orders

import (
	"time"
)

type OrderState int32

const (
	OrderStateUnknown       OrderState = 0
	OrderStateInProgress    OrderState = 1
	OrderStateAwaitsPayment OrderState = 2
	OrderStateCancelled     OrderState = 3
	OrderStateFulfilled     OrderState = 4
	OrderStateRefunded      OrderState = 5
)

type OrderType int32

func (o OrderType) String() string {
	result, ok := orderTypeToString[o]
	if !ok {
		return orderTypeToString[OrderTypeUnknown]
	}
	return result
}

const (
	OrderTypeUnknown  OrderType = 0
	OrderTypeHotel    OrderType = 1
	OrderTypeAvia     OrderType = 2
	OrderTypeTrain    OrderType = 3
	OrderTypeBus      OrderType = 4
	OrderTypeSuburban OrderType = 6 // https://a.yandex-team.ru/arc/trunk/arcadia/travel/orders/proto/commons.proto?rev=r7795752#L55
)

var orderTypeToString = map[OrderType]string{
	OrderTypeUnknown:  "unknown",
	OrderTypeHotel:    "hotel",
	OrderTypeAvia:     "avia",
	OrderTypeTrain:    "train",
	OrderTypeBus:      "bus",
	OrderTypeSuburban: "suburban",
}

type OrderInfo struct {
	ID              string
	Owner           *UserInfo
	Type            OrderType
	State           OrderState
	CorrelationID   string
	AviaOrderItems  []*AviaOrderItem
	TrainOrderItems []*TrainOrderItem
	HotelOrderItems []*HotelOrderItem
}

func (o *OrderInfo) GetEmail() string {
	if o.Owner != nil {
		return o.Owner.Email
	}
	return ""
}

func (o *OrderInfo) GetPassportID() string {
	if o.Owner != nil {
		return o.Owner.PassportID
	}
	return ""
}

type UserInfo struct {
	Login              string
	YandexUID          string
	PassportID         string
	Email              string
	Phone              string
	IP                 string
	AllowsSubscription bool
	GeoID              int32
}

type TrainOrderItem struct {
	TrainNumber      string
	CarNumber        string
	DepartureStation string
	ArrivalStation   string
	DepartureTime    *time.Time
	ArrivalTime      *time.Time
}

type HotelOrderItem struct {
	HotelName    string
	HotelAddress string
	HotelPhone   string
	CheckInDate  *time.Time
	CheckOutDate *time.Time
	GeoRegions   []*GeoRegion
}

type GeoRegion struct {
	GeoID int64
	Type  int64
}

type AviaOrderItem struct {
	OriginDestinations []*AviaOriginDestination
}

type AviaOriginDestination struct {
	DepartureStation string
	ArrivalStation   string
	Segments         []*AviaSegment
}

type AviaSegment struct {
	DepartureStation      string
	DepartureDate         *time.Time
	DepartureDatetime     *time.Time
	ArrivalStation        string
	ArrivalDate           *time.Time
	ArrivalDatetime       *time.Time
	MarketingTitle        *AviaFlightTitle
	OperatingTitle        *AviaFlightTitle
	FlightDurationMinutes int32
}

type AviaFlightTitle struct {
	AirlineID    string
	FlightNumber string
}
