package orders

import (
	"fmt"

	"a.yandex-team.ru/travel/notifier/internal/protoutils"
	"a.yandex-team.ru/travel/orders/proto/services/orders"
	ordersapi "a.yandex-team.ru/travel/orders/proto/services/orders/basic_info/v1"
)

type ProtoToOrderInfoMapper struct {
}

func NewProtoToOrderInfoMapper() *ProtoToOrderInfoMapper {
	return &ProtoToOrderInfoMapper{}
}

func (m *ProtoToOrderInfoMapper) Map(orderInfoProto *ordersapi.BasicOrderInfo) (*OrderInfo, error) {
	if orderInfoProto == nil {
		return nil, nil
	}
	aviaOrderIems, err := m.mapAviaOrderItems(orderInfoProto.AviaOrderItems)
	if err != nil {
		return nil, err
	}
	trainOrderIems, correlationID, err := m.mapTrainItems(orderInfoProto.TrainOrderItems)
	if err != nil {
		return nil, err
	}
	hotelOrderIems, err := m.mapHotelItems(orderInfoProto.HotelOrderItems)
	if err != nil {
		return nil, err
	}
	return &OrderInfo{
		ID:              orderInfoProto.Id,
		Owner:           m.mapOwner(orderInfoProto.Owner),
		Type:            OrderType(orderInfoProto.Type),
		State:           OrderState(orderInfoProto.State),
		CorrelationID:   correlationID,
		AviaOrderItems:  aviaOrderIems,
		TrainOrderItems: trainOrderIems,
		HotelOrderItems: hotelOrderIems,
	}, nil
}

func (m *ProtoToOrderInfoMapper) mapOwner(owner *orders.TUserInfo) *UserInfo {
	if owner == nil {
		return nil
	}
	return &UserInfo{
		Login:              owner.Login,
		YandexUID:          owner.YandexUid,
		PassportID:         owner.PassportId,
		Email:              owner.Email,
		Phone:              owner.Phone,
		IP:                 owner.Ip,
		AllowsSubscription: owner.AllowsSubscription,
		GeoID:              owner.GeoId,
	}
}

func (m *ProtoToOrderInfoMapper) mapAviaOrderItems(itemsProto []*ordersapi.AviaOrderItem) ([]*AviaOrderItem, error) {
	if itemsProto == nil {
		return nil, nil
	}
	result := make([]*AviaOrderItem, 0, len(itemsProto))
	for _, itemProto := range itemsProto {
		item, err := m.mapAviaOrderItem(itemProto)
		if err != nil {
			return nil, err
		}
		result = append(result, item)
	}
	return result, nil
}

func (m *ProtoToOrderInfoMapper) mapAviaOrderItem(itemProto *ordersapi.AviaOrderItem) (*AviaOrderItem, error) {
	if itemProto == nil {
		return nil, nil
	}
	item := &AviaOrderItem{
		OriginDestinations: nil,
	}
	if itemProto.OriginDestinations == nil {
		return item, nil
	}
	item.OriginDestinations = make([]*AviaOriginDestination, 0, len(itemProto.OriginDestinations))
	for _, originDestinationProto := range itemProto.OriginDestinations {
		originDestination, err := m.mapAviaOriginDestination(originDestinationProto)
		if err != nil {
			return nil, err
		}
		item.OriginDestinations = append(item.OriginDestinations, originDestination)
	}
	return item, nil
}

func (m *ProtoToOrderInfoMapper) mapAviaOriginDestination(originDestinationProto *ordersapi.AviaOriginDestination) (
	*AviaOriginDestination,
	error,
) {
	if originDestinationProto == nil {
		return nil, nil
	}
	originDestination := &AviaOriginDestination{
		DepartureStation: originDestinationProto.DepartureStation,
		ArrivalStation:   originDestinationProto.ArrivalStation,
		Segments:         nil,
	}
	if originDestinationProto.Segments == nil {
		return originDestination, nil
	}
	originDestination.Segments = make([]*AviaSegment, 0, len(originDestinationProto.Segments))
	for _, segmentProto := range originDestinationProto.Segments {
		segment, err := m.mapAviaSegment(segmentProto)
		if err != nil {
			return nil, err
		}
		originDestination.Segments = append(originDestination.Segments, segment)
	}
	return originDestination, nil
}

func (m *ProtoToOrderInfoMapper) mapAviaSegment(segmentProto *ordersapi.AviaSegment) (*AviaSegment, error) {
	if segmentProto == nil {
		return nil, nil
	}
	departureDatetime, err := protoutils.MapTimestamp(segmentProto.DepartureDatetime)
	if err != nil {
		return nil, err
	}
	arrivalDatetime, err := protoutils.MapTimestamp(segmentProto.ArrivalDatetime)
	if err != nil {
		return nil, err
	}
	return &AviaSegment{
		DepartureStation:      segmentProto.DepartureStation,
		DepartureDate:         protoutils.MapDate(segmentProto.DepartureDate),
		DepartureDatetime:     departureDatetime,
		ArrivalStation:        segmentProto.ArrivalStation,
		ArrivalDate:           protoutils.MapDate(segmentProto.ArrivalDate),
		ArrivalDatetime:       arrivalDatetime,
		MarketingTitle:        m.mapAviaFlightTitle(segmentProto.MarketingTitle),
		OperatingTitle:        m.mapAviaFlightTitle(segmentProto.OperatingTitle),
		FlightDurationMinutes: segmentProto.FlightDurationMinutes,
	}, nil
}

func (m *ProtoToOrderInfoMapper) mapAviaFlightTitle(titleProto *ordersapi.AviaFlightTitle) *AviaFlightTitle {
	if titleProto == nil {
		return nil
	}
	return &AviaFlightTitle{
		AirlineID:    titleProto.AirlineId,
		FlightNumber: titleProto.FlightNumber,
	}
}

func (m *ProtoToOrderInfoMapper) mapTrainItems(itemsProto []*ordersapi.TrainOrderItem) ([]*TrainOrderItem, string, error) {
	if itemsProto == nil {
		return nil, "", nil
	}
	result := make([]*TrainOrderItem, 0, len(itemsProto))
	correlationID := ""
	for _, itemProto := range itemsProto {
		item, err := m.mapTrainOrderItem(itemProto)
		if err != nil {
			return nil, "", err
		}
		result = append(result, item)
		if correlationID != "" && correlationID != itemProto.CorrelationId {
			return nil, "", fmt.Errorf("order contains more than one value of the CorrelationID")
		}
		correlationID = itemProto.CorrelationId
	}
	return result, correlationID, nil
}

func (m *ProtoToOrderInfoMapper) mapTrainOrderItem(itemProto *ordersapi.TrainOrderItem) (*TrainOrderItem, error) {
	if itemProto == nil {
		return nil, nil
	}
	departureTime, err := protoutils.MapTimestamp(itemProto.DepartureTime)
	if err != nil {
		return nil, err
	}
	arrivalTime, err := protoutils.MapTimestamp(itemProto.ArrivalTime)
	if err != nil {
		return nil, err
	}

	return &TrainOrderItem{
		TrainNumber:      itemProto.TrainNumber,
		CarNumber:        itemProto.CarNumber,
		DepartureStation: itemProto.DepartureStation,
		ArrivalStation:   itemProto.ArrivalStation,
		DepartureTime:    departureTime,
		ArrivalTime:      arrivalTime,
	}, nil
}

func (m *ProtoToOrderInfoMapper) mapHotelItems(itemsProto []*ordersapi.HotelOrderItem) ([]*HotelOrderItem, error) {
	if itemsProto == nil {
		return nil, nil
	}
	result := make([]*HotelOrderItem, 0, len(itemsProto))
	for _, itemProto := range itemsProto {
		item, err := m.mapHotelOrderItem(itemProto)
		if err != nil {
			return nil, err
		}
		result = append(result, item)
	}
	return result, nil
}

func (m *ProtoToOrderInfoMapper) mapHotelOrderItem(itemProto *ordersapi.HotelOrderItem) (*HotelOrderItem, error) {
	if itemProto == nil {
		return nil, nil
	}
	return &HotelOrderItem{
		HotelName:    itemProto.HotelName,
		HotelAddress: itemProto.HotelAddress,
		HotelPhone:   itemProto.HotelPhone,
		CheckInDate:  protoutils.MapDate(itemProto.CheckInDate),
		CheckOutDate: protoutils.MapDate(itemProto.CheckOutDate),
		GeoRegions:   m.mapGeoRegions(itemProto.GeoRegions),
	}, nil
}

func (m *ProtoToOrderInfoMapper) mapGeoRegions(geoRegionsProto []*ordersapi.GeoRegion) []*GeoRegion {
	if geoRegionsProto == nil {
		return nil
	}
	geoRegions := make([]*GeoRegion, 0, len(geoRegionsProto))
	for _, geoRegionProto := range geoRegionsProto {
		geoRegions = append(
			geoRegions, &GeoRegion{
				GeoID: geoRegionProto.GeoId,
				Type:  geoRegionProto.Type,
			},
		)
	}
	return geoRegions
}
