package orders

import (
	"fmt"

	"a.yandex-team.ru/travel/notifier/internal/models"
)

func (o *OrderInfo) ToOrder() (order models.Order, orderErr error) {
	order = models.Order{
		ID:            o.ID,
		State:         models.OrderIncomplete,
		CorrelationID: o.CorrelationID,
	}

	switch o.State {
	case OrderStateCancelled:
		order.State = models.OrderCancelled
	case OrderStateRefunded:
		order.State = models.OrderCancelled
	case OrderStateFulfilled:
		order.WasFulfilled = true
		order.State = models.OrderFulfilled
	}

	switch o.Type {
	case OrderTypeAvia:
		if orderErr = processAviaOrder(o.AviaOrderItems, &order); orderErr != nil {
			return
		}
	case OrderTypeHotel:
		if orderErr = processHotelOrder(o.HotelOrderItems, &order); orderErr != nil {
			return
		}
	case OrderTypeTrain:
		if orderErr = processTrainOrder(o.TrainOrderItems, &order); orderErr != nil {
			return
		}
	default:
		orderErr = &errUnknownOrderTypeImpl{fmt.Sprintf("unknown order type: %v %s", o.Type, o.ID)}
		return
	}
	return
}

func processAviaOrder(aviaOrderItems []*AviaOrderItem, order *models.Order) error {
	order.Type = models.OrderAvia
	if len(aviaOrderItems) == 0 {
		return &errInvalidOrderImpl{"no avia order items for the avia-typed order", models.OrderAvia}
	}
	if aviaOrderItems[0] == nil {
		return &errInvalidOrderImpl{"first avia order item is nil", models.OrderAvia}
	}
	aviaItem := aviaOrderItems[0]
	if len(aviaItem.OriginDestinations) == 0 {
		return &errInvalidOrderImpl{"avia order has no origin-destination entries", models.OrderAvia}
	}
	originDestination := aviaItem.OriginDestinations[0]
	if len(originDestination.Segments) == 0 {
		return &errInvalidOrderImpl{"avia order's origin-destination has no segments", models.OrderAvia}
	}
	firstSegment := originDestination.Segments[0]
	if firstSegment.DepartureDatetime == nil {
		return &errInvalidOrderImpl{"avia order's first segment has no departure date/time", models.OrderAvia}
	}
	for _, segment := range originDestination.Segments {
		if segment.ArrivalDatetime != nil {
			order.ArrivalDate = *segment.ArrivalDatetime
		}
	}
	order.StartDate = *firstSegment.DepartureDatetime
	if order.ArrivalDate.IsZero() {
		order.ArrivalDate = order.StartDate
	}
	return nil
}

func processHotelOrder(hotelOrderItems []*HotelOrderItem, order *models.Order) error {
	order.Type = models.OrderHotel
	if len(hotelOrderItems) == 0 {
		return &errInvalidOrderImpl{"no hotel order items for the hotel-typed order", models.OrderHotel}
	}
	if hotelOrderItems[0] == nil {
		return &errInvalidOrderImpl{"first hotel order item is nil", models.OrderHotel}
	}
	hotelItem := hotelOrderItems[0]
	if hotelItem.CheckInDate == nil {
		return &errInvalidOrderImpl{"hotel order's item has no check-in date", models.OrderHotel}
	}
	order.StartDate = *hotelItem.CheckInDate
	order.ArrivalDate = order.StartDate
	return nil
}

func processTrainOrder(trainOrderItems []*TrainOrderItem, order *models.Order) error {
	order.Type = models.OrderTrain
	if len(trainOrderItems) == 0 {
		return &errInvalidOrderImpl{"no train order items for the train-typed order", models.OrderTrain}
	}
	if trainOrderItems[0] == nil {
		return &errInvalidOrderImpl{"first train order item is nil", models.OrderTrain}
	}
	trainItem := trainOrderItems[0]
	if trainItem.DepartureTime == nil {
		return &errInvalidOrderImpl{"train order's item has no departure time", models.OrderTrain}
	}
	order.StartDate = *trainItem.DepartureTime
	if trainItem.ArrivalTime != nil {
		order.ArrivalDate = *trainItem.ArrivalTime
	} else if !order.StartDate.IsZero() {
		order.ArrivalDate = order.StartDate
	} else {
		return &errInvalidOrderImpl{"first train order item does not have an arrival time", models.OrderTrain}
	}

	return nil
}
