package orders

import (
	"context"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	balancerconfig "a.yandex-team.ru/travel/library/go/grpcutil/client/balancer_config"
	"a.yandex-team.ru/travel/library/go/grpcutil/client/ypresolver"
	"a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/notifier/internal/configtypes"
	ordersapi "a.yandex-team.ru/travel/orders/proto/services/orders/basic_info/v1"
	"a.yandex-team.ru/travel/orders/proto/services/promo"
)

var DefaultOrdersAppTimeout = 60 * time.Second

type OrderInfoMapper interface {
	Map(orderInfoProto *ordersapi.BasicOrderInfo) (*OrderInfo, error)
}

type Client struct {
	tvmInterceptor    grpc.UnaryClientInterceptor
	ordersAppContacts OrdersAppConfig
	grpcClientConfig  configtypes.GrpcClientConfig
	orderInfoMapper   OrderInfoMapper
	logger            log.Logger
}

func NewClient(
	logger log.Logger,
	selfTvmID uint32,
	ordersAppContacts OrdersAppConfig,
	grpcClientConfig configtypes.GrpcClientConfig,
	orderInfoMapper OrderInfoMapper,
) *Client {
	tvmHelper := tvm.NewDeployTvmHelper(
		logger, &tvm.TvmHelperConfig{
			SelfID:    selfTvmID,
			WhiteList: []uint32{},
		},
	)
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(ordersAppContacts.TvmID)
	if tvmInterceptor == nil {
		return nil
	}

	return &Client{
		tvmInterceptor:    tvmInterceptor,
		ordersAppContacts: ordersAppContacts,
		grpcClientConfig:  grpcClientConfig,
		orderInfoMapper:   orderInfoMapper,
		logger:            logger,
	}
}

func (oc *Client) GetOrderInfoByID(ctx context.Context, orderID string) (orderInfo *OrderInfo, orderErr error) {
	response, err := oc.sendRequest(ctx, orderID)
	if err != nil {
		orderErr = err
		return
	}

	if response == nil {
		orderErr = fmt.Errorf("orders client returned nil")
		return
	}

	return oc.orderInfoMapper.Map(response.OrderInfo)
}

func (oc *Client) sendRequest(ctx context.Context, orderID string) (response *ordersapi.GetBasicOrderResponse, orderErr error) {
	ctx, cancel := context.WithTimeout(ctx, DefaultOrdersAppTimeout)
	defer cancel()

	gConn, err := oc.prepareConnection(ctx)
	if err != nil {
		return
	}
	defer gConn.Close()

	ordersClient := ordersapi.NewBasicOrderInfoAPIClient(gConn)
	req := ordersapi.GetBasicOrderRequest{
		OneOfOrderIds: &ordersapi.GetBasicOrderRequest_OrderId{OrderId: orderID},
	}

	response, err = ordersClient.GetOrder(ctx, &req)
	if err != nil {
		orderErr = fmt.Errorf("orders app returned an error: %w", err)
		return
	}
	return
}

func (oc *Client) CreatePromoCode(
	ctx context.Context,
	promoCampaignID string,
	code string,
	validTill time.Time,
	promoActionDetails *promo.TGetPromoActionResp,
) (
	*promo.TCreatePromoCodeRsp, error,
) {
	ctx, cancel := context.WithTimeout(ctx, DefaultOrdersAppTimeout)
	defer cancel()

	gConn, err := oc.prepareConnection(ctx)
	if err != nil {
		return nil, err
	}
	defer gConn.Close()

	ordersClient := promo.NewPromoCodesOperatorManagementInterfaceV1Client(gConn)

	createPromoCodeReq := promo.TCreatePromoCodeReq{
		PromoActionId:  promoCampaignID,
		Code:           code,
		Nominal:        promoActionDetails.GetGenerationConfig().GetNominal(),
		NominalType:    promoActionDetails.GetGenerationConfig().GetNominalType(),
		MaxUsageCount:  promoActionDetails.GetGenerationConfig().GetMaxUsagePerUser(),
		MaxActivations: promoActionDetails.GetGenerationConfig().GetMaxActivations(),
		ValidFrom:      promoActionDetails.GetActionValidFrom(),
		ValidTill:      timestamppb.New(validTill),
	}

	oc.logger.Debug("requesting CreatePromoCode", log.Any("req", &createPromoCodeReq))
	response, err := ordersClient.CreatePromoCode(ctx, &createPromoCodeReq)
	if err != nil {
		return nil, fmt.Errorf("orders app returned an error: %w", err)
	}
	oc.logger.Debug("response of CreatePromoCode", log.Any("rsp", &response))
	return response, nil
}

func (oc *Client) GetPromoActionDetails(ctx context.Context, promoCampaignID string) (
	response *promo.TGetPromoActionResp,
	orderErr error,
) {
	ctx, cancel := context.WithTimeout(ctx, DefaultOrdersAppTimeout)
	defer cancel()

	gConn, err := oc.prepareConnection(ctx)
	if err != nil {
		return
	}
	defer gConn.Close()

	ordersClient := promo.NewPromoCodesOperatorManagementInterfaceV1Client(gConn)
	req := promo.TGetPromoActionReq{
		PromoActionId: promoCampaignID,
	}

	oc.logger.Debug("requesting GetPromoActionDetails", log.Any("req", &req))
	response, err = ordersClient.GetPromoActionDetails(ctx, &req)
	if err != nil {
		orderErr = fmt.Errorf("orders app returned an error: %w", err)
		return
	}
	oc.logger.Debug("response of GetPromoActionDetails", log.Any("rsp", &response))
	return
}

func (oc *Client) PromoCodeActivationAvailable(ctx context.Context, code string) (
	response *promo.TPromoCodeActivationAvailableResp,
	orderErr error,
) {
	ctx, cancel := context.WithTimeout(ctx, DefaultOrdersAppTimeout)
	defer cancel()

	gConn, err := oc.prepareConnection(ctx)
	if err != nil {
		return
	}
	defer gConn.Close()

	ordersClient := promo.NewPromoCodesOperatorManagementInterfaceV1Client(gConn)
	req := promo.TPromoCodeActivationAvailableReq{
		Code: code,
	}

	oc.logger.Debug("requesting PromoCodeActivationAvailable", log.Any("req", &req))
	response, err = ordersClient.PromoCodeActivationAvailable(ctx, &req)
	if err != nil {
		orderErr = fmt.Errorf("orders app returned an error: %w", err)
		return
	}
	oc.logger.Debug("response of PromoCodeActivationAvailable", log.Any("rsp", &response))
	return
}

func (oc *Client) prepareConnection(ctx context.Context) (conn *grpc.ClientConn, err error) {
	resolver := ypresolver.NewYPResolverBuilder(
		ypresolver.WithClusters(oc.grpcClientConfig.Clusters),
		ypresolver.WithLogger(oc.logger.Structured()),
	)
	gConn, err := grpc.DialContext(
		ctx,
		ypresolver.BuildServiceFQDN(oc.ordersAppContacts.YPlannerID),
		grpc.WithResolvers(resolver),
		grpc.WithDefaultServiceConfig(balancerconfig.RoundRobin()),
		grpc.WithInsecure(),
		grpc.WithBlock(),
		grpc.WithUnaryInterceptor(oc.tvmInterceptor),
	)
	if err != nil {
		err = fmt.Errorf("unable to connect to the orders app: %w", err)
		return nil, err
	}
	return gConn, nil
}
