package processor

import (
	"fmt"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/jonboulle/clockwork"
	"golang.yandex/hasql"
	gormlogger "gorm.io/gorm/logger"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/contentadmin"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/sender"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/externalhttp"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/pgclient"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/audioguide"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/events"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/hotels"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/weather"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/logging/renderlog"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
	"a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/scheduler"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/travelapi"
)

func BuildPGClient(
	logger log.Logger,
) (*pgclient.PGClient, error) {
	logLevel := gormlogger.Silent
	if Cfg.IsDevelopment() {
		logLevel = gormlogger.Info
	}
	return pgclient.NewPGClient(
		Cfg.Database.Hosts,
		Cfg.Database.Port,
		Cfg.Database.Name,
		Cfg.Database.User,
		Cfg.Database.Password,
		pgclient.DefaultInitTimeout,
		[]pgclient.ClientOption{pgclient.WithOnCheckedNode(database.OnCheckedNode), pgclient.WithLogLevel(logLevel)},
		append(
			pgclient.DefaultClusterOptions, hasql.WithUpdateTimeout(Cfg.Database.HostsUpdateTimeout), hasql.WithTracer(
				hasql.Tracer{
					UpdatedNodes: database.OnUpdatedNodes,
					NodeDead:     database.OnDeadNode(logger),
				},
			),
		),
	)
}

func BuildNotificationsRepository(pgClient *pgclient.PGClient) *database.NotificationsRepository {
	gormDebugMode := Cfg.IsDevelopment()
	notificationsRepository := database.NewNotificationsRepository(pgClient, gormDebugMode)
	return notificationsRepository
}

func BuildBetterPriceSubscriptionRepository(pgClient *pgclient.PGClient) *database.BetterPriceSubscriptionsRepository {
	gormDebugMode := Cfg.IsDevelopment()
	betterPriceSubscriptionsRepository := database.NewBetterPriceSubscriptionsRepository(pgClient, gormDebugMode)
	return betterPriceSubscriptionsRepository
}

func BuildSenderClient(logger log.Logger) *sender.HTTPClient {
	return sender.NewHTTPClient(
		logger, sender.Config{
			HTTPClient: &http.Client{
				Timeout: Cfg.Sender.Timeout,
			},
			AuthKey:                Cfg.Sender.AuthKey,
			SenderURL:              Cfg.Sender.URL,
			Account:                Cfg.Sender.Account,
			AllowInactiveCampaigns: Cfg.Sender.AllowInactiveCampaigns,
		},
		sender.WithBackOffPolicy(
			func() backoff.BackOff {
				return &backoff.ExponentialBackOff{
					InitialInterval:     100 * time.Millisecond,
					RandomizationFactor: backoff.DefaultRandomizationFactor,
					Multiplier:          backoff.DefaultMultiplier,
					MaxInterval:         500 * time.Millisecond,
					MaxElapsedTime:      2 * Cfg.Sender.Timeout,
					Stop:                backoff.Stop,
					Clock:               backoff.SystemClock,
				}
			},
		),
		sender.WithRequestTimeout(Cfg.Sender.Timeout),
	)
}

func BuildOrdersClient(logger log.Logger) *orders.Client {
	return orders.NewClient(logger, Cfg.Tvm.SelfAppID, Cfg.OrdersApp, Cfg.GrpcClientConfig, orders.NewProtoToOrderInfoMapper())
}

func BuildPretripProcessor(
	logger log.Logger,
	notificationsRepository *database.NotificationsRepository,
	recipientsRepository *database.RecipientsRepository,
	ordersRepository *database.OrdersRepository,
	senderClient *sender.HTTPClient,
	ordersClient *orders.Client,
	tvmClient *tvmtool.Client,
	dictsRegistry *dicts.Registry,
	notificationsBuilder scheduling.NotificationsBuilder,
	notificationsScheduler *scheduler.Service,
	clock clockwork.Clock,
	rollOutService *rollout.Service,
	unifiedAgentClient unifiedagent.Client,
	settlementDataProvider *extractors.SettlementDataProvider,
	stationDataProvider *extractors.StationDataProvider,
) (*pretrip.Processor, error) {
	rendererClient := renderer.NewHTTPClient(
		renderer.Config{
			RendererURL: Cfg.Renderer.URL,
			HTTPClient: &http.Client{
				Timeout: Cfg.Renderer.Timeout,
			},
		},
		renderer.WithBackOffPolicy(
			func() backoff.BackOff {
				return &backoff.ExponentialBackOff{
					InitialInterval:     100 * time.Millisecond,
					RandomizationFactor: backoff.DefaultRandomizationFactor,
					Multiplier:          backoff.DefaultMultiplier,
					MaxInterval:         500 * time.Millisecond,
					MaxElapsedTime:      10 * time.Second,
					Clock:               backoff.SystemClock,
					Stop:                backoff.Stop,
				}
			},
		),
		renderer.WithRequestTimeout(Cfg.Renderer.Timeout),
	)

	contentAdminClient := contentadmin.NewContentAdminClient(logger, Cfg.Tvm.SelfAppID, Cfg.ContentAdmin.ToLibConfig())
	if contentAdminClient == nil {
		return nil, fmt.Errorf("failed to create a client to the content admin service")
	}

	iziTravelClient := audioguide.NewAudioGuidesProvider(
		externalhttp.NewHTTPClient(
			Cfg.Pretrip.BlockProviders.AudioGuidesProvider,
			&http.Client{
				Timeout: Cfg.Pretrip.BlockProviders.AudioGuidesProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(Cfg.Pretrip.BlockProviders.AudioGuidesProviderTimeout),
		),
	)

	afishaClient := events.NewAfishaEventsClient(
		externalhttp.NewHTTPClient(
			Cfg.Pretrip.BlockProviders.EventsProvider,
			&http.Client{
				Timeout: Cfg.Pretrip.BlockProviders.EventsProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(Cfg.Pretrip.BlockProviders.EventsProviderTimeout),
		),
	)

	weatherClient := weather.NewWeatherProvider(
		externalhttp.NewHTTPClient(
			Cfg.Pretrip.BlockProviders.WeatherProvider,
			&http.Client{
				Timeout: Cfg.Pretrip.BlockProviders.WeatherProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(Cfg.Pretrip.BlockProviders.WeatherProviderTimeout),
		),
	)

	hotelsClient := hotels.NewHotelsClient(
		externalhttp.NewHTTPClient(
			Cfg.Pretrip.BlockProviders.HotelsProvider,
			&http.Client{
				Timeout: Cfg.Pretrip.BlockProviders.HotelsProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(Cfg.Pretrip.BlockProviders.HotelsProviderTimeout),
		),
	)
	additionalOrderInfoClient := travelapi.NewAdditionalOrderInfoClient(
		externalhttp.NewHTTPClient(
			Cfg.Pretrip.BlockProviders.AdditionalOrderInfo,
			&http.Client{
				Timeout: Cfg.Pretrip.BlockProviders.AdditionalOrderInfoProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(Cfg.Pretrip.BlockProviders.AdditionalOrderInfoProviderTimeout),
		),
	)

	routePointsFromOrderExtractor := BuildOrderDestinationExtractor(dictsRegistry)
	unsubscribeLinkGenerator := subscriptions.NewUnsubscribeLinkGenerator(Cfg.TravelPortalURL, Cfg.Subscriptions.UnsubscribePath)

	pretripBlocksCollector := blocks.NewBlocksCollector(
		logger,
		pretrip.GetDefaultBlockProviders(
			Cfg.TravelPortalURL,
			contentAdminClient,
			iziTravelClient,
			afishaClient,
			weatherClient,
			routePointsFromOrderExtractor,
			unsubscribeLinkGenerator,
			settlementDataProvider,
			stationDataProvider,
			dictsRegistry.GetTimeZonesRepository(),
			hotelsClient,
			additionalOrderInfoClient,
		)...,
	)

	pretripProcessor := pretrip.NewProcessor(
		logger,
		notificationsRepository,
		recipientsRepository,
		ordersRepository,
		rendererClient,
		senderClient,
		ordersClient,
		pretripBlocksCollector,
		Cfg.Pretrip,
		clockwork.NewRealClock(),
		routePointsFromOrderExtractor,
		pretrip.NewNotificationDeadlineHandler(notificationsBuilder, notificationsScheduler, clock),
		rollOutService,
		renderlog.NewLogger(unifiedAgentClient),
		settlementDataProvider,
	)
	return pretripProcessor, nil
}

func BuildOrderDestinationExtractor(dictsRegistry *dicts.Registry) *extractors.RoutePointsFromOrderExtractor {
	stationIDToSettlementIDMapper := extractors.NewStationIDToSettlementIDMapper(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
	)

	return extractors.NewRoutePointsFromOrderExtractor(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
		stationIDToSettlementIDMapper,
	)
}
