package processor

import (
	"time"

	dynamicdicts "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	"a.yandex-team.ru/travel/library/go/logging"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/configtypes"
	"a.yandex-team.ru/travel/notifier/internal/contentadmin"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/polling"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip"
	"a.yandex-team.ru/travel/notifier/internal/service/processing"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/queueobserver"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	"a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts"
)

type DebugConfig struct {
	Email               string `config:"DEBUG_EMAIL"`
	OrderID             string `config:"DEBUG_ORDER_ID"`
	NotificationSubtype string `config:"DEBUG_SUBTYPE"`
}

type TvmConfig struct {
	Enabled   bool     `config:"TVM_ENABLED" yaml:"enabled"`
	WhiteList []uint32 `config:"TVM_WHITELIST" yaml:"whitelist"`
	SelfAppID uint32   `config:"TVM_SELFAPP_ID,required" yaml:"selfappid"`
}

type Config struct {
	Tvm              TvmConfig
	EnvType          string `config:"YENV_TYPE,required" yaml:"yenv_type"`
	TravelPortalURL  string `config:"TRAVEL_PORTAL_URL" yaml:"travel_portal_url"`
	Metrics          metricsserver.MetricsConfig
	Logging          logging.Config
	Database         database.Config
	OrdersApp        orders.OrdersAppConfig
	ContentAdmin     contentadmin.Config
	Pretrip          pretrip.ProcessingConfig
	Polling          polling.Config
	Shutdown         shutdown.Config
	Readiness        readiness.Config
	Sender           SenderConfig
	Renderer         RendererConfig
	Dicts            dicts.Config
	YtDicts          ytdicts.Config `yaml:"yt_dicts"`
	Processing       processing.Config
	DynamicResources dynamicdicts.Config `yaml:"dynamic_resources"`
	Debug            DebugConfig
	RollOut          rollout.Config `yaml:"roll_out"`
	Profiling        profiling.Config
	UnifiedAgent     unifiedagent.ClientConfig    `yaml:"unified_agent"`
	QueueObserving   queueobserver.Config         `yaml:"queue_observing"`
	GrpcClientConfig configtypes.GrpcClientConfig `yaml:"grpc_client"`
	Subscriptions    subscriptions.Config
}

func (c Config) IsDevelopment() bool {
	return c.EnvType == "development"
}

func (c Config) IsTesting() bool {
	return c.EnvType == "testing"
}

type SenderConfig struct {
	Timeout                time.Duration `config:"SENDER_TIMEOUT" yaml:"timeout"`
	AuthKey                string        `config:"SENDER_AUTHKEY" yaml:"authkey"`
	URL                    string        `config:"SENDER_URL" yaml:"url"`
	Account                string        `config:"SENDER_ACCOUNT" yaml:"account"`
	AllowInactiveCampaigns bool          `config:"SENDER_ALLOW_INACTIVE_CAMPAIGNS" yaml:"allow_inactive_campaigns"`
}

type RendererConfig struct {
	Timeout time.Duration `config:"RENDERER_TIMEOUT" yaml:"timeout"`
	URL     string        `config:"RENDERER_URL" yaml:"url"`
}

var (
	DefaultMetricsConfig = metricsserver.MetricsConfig{
		Addr:                       "[::]:8383",
		PerfMetricsRefreshInterval: 5 * time.Second,
		PerfMetricsPrefix:          "perf",
	}

	DefaultSenderConfig = SenderConfig{
		Timeout:                5 * time.Second,
		URL:                    "https://test.sender.yandex-team.ru",
		Account:                "travel",
		AllowInactiveCampaigns: false,
	}

	DefaultRendererConfig = RendererConfig{
		Timeout: 5 * time.Second,
		URL:     "https://travel-tools-test.yandex.net/renderer/getStructuredMail",
	}

	Cfg = Config{
		EnvType:          "development",
		Metrics:          DefaultMetricsConfig,
		Logging:          logging.DefaultConfig,
		Tvm:              TvmConfig{},
		Database:         database.DefaultConfig,
		Pretrip:          pretrip.DefaultProcessingConfig,
		Polling:          polling.DefaultConfig,
		Shutdown:         shutdown.DefaultConfig,
		Readiness:        readiness.DefaultConfig,
		Sender:           DefaultSenderConfig,
		Renderer:         DefaultRendererConfig,
		Dicts:            dicts.DefaultConfig,
		Processing:       processing.DefaultConfig,
		Subscriptions:    subscriptions.DefaultConfig,
		ContentAdmin:     contentadmin.DefaultConfig,
		TravelPortalURL:  "https://travel-test.yandex.ru",
		DynamicResources: dynamicdicts.DefaultConfig,
		RollOut:          rollout.DefaultConfig,
		Profiling:        profiling.DefaultConfig,
		UnifiedAgent:     unifiedagent.DefaultClientConfig,
		QueueObserving:   queueobserver.DefaultConfig,
	}
)
