package rollout

import (
	"fmt"
	"regexp"
)

type EmailRules struct {
	AlwaysAllowed []EmailRule `yaml:"always_allowed"`
	Banned        []EmailRule `yaml:"banned"`
	Allowed       []EmailRule `yaml:"allowed"`
}

type EmailRule struct {
	regexp *regexp.Regexp
}

func NewEmailRule(expression string) (*EmailRule, error) {
	regExp, err := regexp.Compile(expression)
	if err != nil {
		return nil, err
	}
	return &EmailRule{regexp: regExp}, nil
}

func (r *EmailRule) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var expression string
	err := unmarshal(&expression)
	if err != nil {
		return err
	}

	rule, err := NewEmailRule(expression)
	if err != nil {
		return fmt.Errorf("invalid regexp for EmailRule: %w", err)
	}
	*r = *rule
	return nil
}

func (r *EmailRule) Apply(email string) bool {
	return r.regexp.MatchString(email)
}
