package onlineregistration

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
)

const flightTimeFormat = "2006-01-02T15:04"

type NotificationsBuilder struct {
	stationCodesRepository StationCodesRepository
	stationsRepository     StationsRepository
	timeZoneDataProvider   TimeZoneDataProvider
}

func NewNotificationsBuilder(
	stationCodesRepository StationCodesRepository,
	stationsRepository StationsRepository,
	timeZoneDataProvider TimeZoneDataProvider,
) *NotificationsBuilder {
	return &NotificationsBuilder{
		stationCodesRepository: stationCodesRepository,
		stationsRepository:     stationsRepository,
		timeZoneDataProvider:   timeZoneDataProvider,
	}
}

func (b *NotificationsBuilder) Build(orderInfo *orders.OrderInfo, user models.User, now time.Time) ([]models.Notification, error) {
	if orderInfo.Type != orders.OrderTypeAvia {
		return nil, nil
	}
	order, _ := orderInfo.ToOrder()
	result := make([]models.Notification, 0)
	for _, aviaOrderItem := range orderInfo.AviaOrderItems {
		for _, originDestination := range aviaOrderItem.OriginDestinations {
			for _, segment := range originDestination.Segments {
				if !segment.DepartureDatetime.After(now) {
					return nil, nil
				}
				if notification, err := b.buildNotification(&order, user, segment, now); notification != nil {
					result = append(result, *notification)
				} else {
					return nil, err
				}
			}
		}
	}
	return result, nil
}

func (b *NotificationsBuilder) buildNotification(order *models.Order, user models.User, segment *orders.AviaSegment, now time.Time) (*models.Notification, error) {
	if segment == nil {
		return nil, nil
	}
	departureStationID, found := b.stationCodesRepository.GetStationIDByCode(segment.DepartureStation)
	if !found {
		return nil, fmt.Errorf("unknown station code: %s", segment.DepartureStation)
	}
	arrivalStationID, found := b.stationCodesRepository.GetStationIDByCode(segment.ArrivalStation)
	if !found {
		return nil, fmt.Errorf("unknown station code: %s", segment.DepartureStation)
	}
	departureStation, found := b.stationsRepository.Get(int(departureStationID))
	if !found {
		return nil, fmt.Errorf("unknown station id: %d", departureStationID)
	}
	timeZone, found := b.timeZoneDataProvider.Get(int(departureStation.TimeZoneId))
	if !found {
		return nil, fmt.Errorf("unknown timezone id: %d", departureStation.TimeZoneId)
	}
	localDepartureTime, err := time.ParseInLocation(flightTimeFormat, segment.DepartureDatetime.Format(flightTimeFormat), timeZone)
	if err != nil {
		return nil, fmt.Errorf("failed to parse departure time: %w", err)
	}
	onlineRegistrationStart := localDepartureTime.Add(-constants.DayDuration)
	notifyAt := onlineRegistrationStart.UTC()
	if notifyAt.Before(now) {
		notifyAt = now
	}
	deadline := onlineRegistrationStart.Add(constants.DayDuration).UTC()
	payload := b.buildPayload(localDepartureTime, segment, departureStationID, arrivalStationID)
	notification := models.NewNotification(
		notifyAt,
		deadline,
		models.NotificationStatusPlanned,
		models.NotificationTypeOnlineRegistration,
		models.NotificationChannelPullAPI,
		models.DispatchTypePull,
	).WithOrder(*order).WithPayload(payload).WithUser(user)
	return &notification, nil
}

func (b *NotificationsBuilder) buildPayload(localDeparture time.Time, segment *orders.AviaSegment, departureStationID, arrivalStationID int32) []byte {
	model := Payload{LocalDepartureTime: localDeparture.Format(flightTimeFormat)}
	if segment.MarketingTitle != nil {
		model.MarketingFlightNumber = b.buildFlightNumber(segment.MarketingTitle)
	}
	if segment.OperatingTitle != nil {
		model.OperatingFlightNumber = b.buildFlightNumber(segment.OperatingTitle)
	}
	model.DepartureStationID = departureStationID
	model.ArrivalStationID = arrivalStationID
	payload, _ := json.Marshal(model)
	return payload
}

func (b *NotificationsBuilder) buildFlightNumber(title *orders.AviaFlightTitle) string {
	return fmt.Sprintf("%s %s", title.AirlineID, title.FlightNumber)
}
