package onlineregistration

import (
	"context"
	"fmt"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/orders"
)

type NotificationsService struct {
	logger                 log.Logger
	notificationsScheduler NotificationsScheduler
	usersRepository        UsersRepository
	clock                  clockwork.Clock
	notificationsBuilder   *NotificationsBuilder
	config                 Config
}

func NewNotificationsService(
	config Config,
	logger log.Logger,
	notificationsScheduler NotificationsScheduler,
	usersRepository UsersRepository,
	notificationsBuilder *NotificationsBuilder,
	clock clockwork.Clock,
) *NotificationsService {
	return &NotificationsService{
		config:                 config,
		logger:                 logger.WithName("OnlineRegistrationNotificationsService"),
		notificationsScheduler: notificationsScheduler,
		usersRepository:        usersRepository,
		notificationsBuilder:   notificationsBuilder,
		clock:                  clock,
	}
}

func (p *NotificationsService) OnOrderChanged(
	ctx context.Context,
	orderInfo *orders.OrderInfo,
) (result orderchanges.OrderChangedResult) {
	if !p.config.Enabled {
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK}
	}
	ctx = ctxlog.WithFields(ctx, log.String("orderID", orderInfo.ID))
	defer func() {
		if result.Code != "" {
			metrics.GlobalAppMetrics().GetOrCreateCounter(
				onlineRegistrationMetricsPrefix,
				map[string]string{"code": result.Code},
				notPlannedNotificationMetricName,
			).Inc()
		}
		ctxlog.Info(
			ctx,
			p.logger,
			"OnOrderChanged request processed",
			log.String("code", result.Code),
			log.String("message", result.Message),
		)
	}()

	if orderInfo.GetPassportID() == "" {
		ctxlog.Info(ctx, p.logger, "skip order without owner or passport ID")
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK, Code: "no_passport_id"}
	}
	passportID := orderInfo.Owner.PassportID
	ctx = ctxlog.WithFields(ctx, log.String("passportID", passportID))

	_, err := p.notificationsScheduler.CancelPlannedNotificationsByOrderID(ctx, orderInfo.ID, models.NotificationTypeOnlineRegistration, models.DispatchTypePull)
	if err != nil {
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("failed to cancel planned for order %s: %s", orderInfo.ID, err),
			Code:    "cancel_planned_error",
		}
	}

	now := p.clock.Now()
	user, err := p.usersRepository.GetOrCreate(ctx, buildUser(passportID))
	if err != nil {
		p.logger.Error("failed to create a user", log.Error(err), log.String("orderID", orderInfo.ID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusFailure,
			Message: fmt.Sprintf("failed to create a user: %s", err),
			Code:    "recipient_create_error",
		}
	}

	notifications, err := p.notificationsBuilder.Build(orderInfo, *user, now)
	if err != nil {
		ctxlog.Error(ctx, p.logger, "failed to build notifications", log.Error(err))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusFailure,
			Message: fmt.Sprintf("couldn't build notification: %s", err),
			Code:    "build_error",
		}
	}
	if len(notifications) == 0 {
		ctxlog.Info(ctx, p.logger, "0 notifications have been built for order")
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK}
	}
	if _, err := p.notificationsScheduler.ScheduleMany(ctx, notifications, now); err != nil {
		ctxlog.Error(ctx, p.logger, "failed to schedule notifications", log.Error(err))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("couldn't schedule notification: %s", err),
			Code:    "schedule_error",
		}
	}
	ctxlog.Info(
		ctx,
		p.logger,
		"notifications have been successfully scheduled",
		log.Int("notificationsCount", len(notifications)),
	)
	for range notifications {
		metrics.GlobalAppMetrics().GetOrCreateCounter(onlineRegistrationMetricsPrefix, nil, plannedNotificationMetricName).Inc()
	}
	return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK}
}

func buildUser(passportID string) models.User {
	return models.NewUser().WithPassportID(passportID)
}
