package orderchanges

import (
	"context"
	"errors"
	"fmt"
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/orders"
)

type Service struct {
	logger                 log.Logger
	handleChangesFromQueue bool
	subscribers            []OnOrderChangedSubscriber
	orderProvider          OrderProvider
	ordersRepository       OrdersRepository
	orderStatusChecker     OrderStatusChecker
}

func (s *Service) OnOrderChanged(ctx context.Context, request orderchanges.OrderChangedRequest) orderchanges.OrderChangedResult {
	s.logger.Debugf("OrderChangesService.OnOrderChanged called %+v", request)
	return s.notifyAll(ctx, request)
}

func (s *Service) notifyAll(ctx context.Context, request orderchanges.OrderChangedRequest) orderchanges.OrderChangedResult {
	if !s.needHandleRequest(request) {
		message := fmt.Sprintf("skip changes from source %s", string(request.Source))
		s.logger.Info(message)
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: message,
		}
	}

	notificationResults := make(chan orderchanges.OrderChangedResult, len(s.subscribers))
	orderInfo, err := s.orderProvider.GetOrderInfoByID(ctx, request.OrderID)
	if err != nil {
		s.logger.Error("failed to get order info", log.Error(err), log.String("orderID", request.OrderID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("failed to get order info: %s", err),
			Code:    "order_info_error",
		}
	}
	if result := s.handleOrderInfo(ctx, orderInfo); result != nil {
		return *result
	}
	wg := sync.WaitGroup{}
	wg.Add(len(s.subscribers))
	for _, subscriber := range s.subscribers {
		go s.notify(ctx, &wg, orderInfo, subscriber, notificationResults)
	}
	wg.Wait()
	close(notificationResults)
	return s.buildNotificationResult(notificationResults)
}

func (s *Service) buildNotificationResult(notificationResults chan orderchanges.OrderChangedResult) (result orderchanges.OrderChangedResult) {
	result.Status = orderchanges.OrderChangedStatusOK
	messages := make([]string, 0, len(notificationResults))
	for r := range notificationResults {
		if r.Status > result.Status {
			result.Status = r.Status
		}
		if result.Status != orderchanges.OrderChangedStatusOK {
			messages = append(messages, r.Message)
		}
	}
	result.Message = strings.Join(messages, "\n")
	return result
}

func (s *Service) notify(
	ctx context.Context,
	wg *sync.WaitGroup,
	orderInfo *orders.OrderInfo,
	subscriber OnOrderChangedSubscriber,
	results chan orderchanges.OrderChangedResult,
) {
	defer wg.Done()
	defer func() {
		if r := recover(); r != nil {
			results <- orderchanges.OrderChangedResult{
				Status:  orderchanges.OrderChangedStatusTemporaryError,
				Message: fmt.Sprintf("panic: %s", r),
			}
		}
	}()
	results <- subscriber.OnOrderChanged(ctx, orderInfo)
}

func (s *Service) handleOrderInfo(ctx context.Context, orderInfo *orders.OrderInfo) *orderchanges.OrderChangedResult {
	order, res := mapOrderInfoToOrder(orderInfo)
	if res != nil {
		return res
	}

	email := orderInfo.GetEmail()
	passportID := orderInfo.GetPassportID()
	if len(email) == 0 && len(passportID) == 0 {
		s.logger.Warn("empty email and passport id", log.String("orderID", orderInfo.ID))
		return &orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK, Code: "no_email_no_passport"}
	}

	if isCorrectOrderStatus, err := s.orderStatusChecker.Check(ctx, order); err != nil {
		return &orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("failed to check order: %s", err),
			Code:    "order_check_error",
		}
	} else if !isCorrectOrderStatus {
		return &orderchanges.OrderChangedResult{
			Status: orderchanges.OrderChangedStatusOK,
			Message: fmt.Sprintf(
				"notifications haven't been planned for order %s: Pretrip is not applicable to current order status: %s",
				orderInfo.ID,
				order.State,
			),
			Code: "not_fulfilled_order",
		}
	} else {
		// It's crucial to ensure that this field won't be overridden when fulfilled order becomes cancelled
		order.WasFulfilled = true
		if err := s.ordersRepository.Upsert(ctx, order); err != nil {
			return &orderchanges.OrderChangedResult{
				Status:  orderchanges.OrderChangedStatusTemporaryError,
				Message: fmt.Sprintf("failed to c order: %s", err),
				Code:    "order_upsert_error",
			}
		}
	}
	return nil
}

func (s *Service) needHandleRequest(request orderchanges.OrderChangedRequest) bool {
	return s.handleChangesFromQueue == (request.Source == orderchanges.OrderChangedSourceQueue)
}

func mapOrderInfoToOrder(orderInfo *orders.OrderInfo) (models.Order, *orderchanges.OrderChangedResult) {
	order, err := orderInfo.ToOrder()
	if err == nil {
		return order, nil
	}
	message := fmt.Sprintf("%s: %s", "couldn't parse order info", err)
	var unknownOrderType orders.ErrUnknownOrderType
	if errors.As(err, &unknownOrderType) {
		return order, &orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: message,
			Code:    "unknown_order_type",
		}
	}
	return order, &orderchanges.OrderChangedResult{
		Status:  orderchanges.OrderChangedStatusFailure,
		Message: message,
		Code:    "order_map_error",
	}
}

func NewService(
	l log.Logger,
	handleChangesFromQueue bool,
	orderProvider OrderProvider,
	ordersRepository OrdersRepository,
	orderStatusChecker OrderStatusChecker,
	subscribers ...OnOrderChangedSubscriber,
) *Service {
	return &Service{
		logger:                 l,
		handleChangesFromQueue: handleChangesFromQueue,
		subscribers:            subscribers,
		orderProvider:          orderProvider,
		ordersRepository:       ordersRepository,
		orderStatusChecker:     orderStatusChecker,
	}
}
