package polling

import (
	"context"
	"time"

	"github.com/gofrs/uuid"
	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/notifier/internal/models"
)

type NotificationsGetter interface {
	GetFirstN(ctx context.Context, now time.Time, plannedLimit, postponedLimit, notSentLimit uint) ([]models.Notification, error)
	GetForOrder(ctx context.Context, orderID string, from time.Time, until time.Time, plannedOnly bool) ([]models.Notification, error)
}

type NotificationsFilter interface {
	Filter(uuid.UUID, []models.Notification) []models.Notification
}

type NotificationsProcessor interface {
	Process(ctx context.Context, notifications []models.Notification, iterationID uuid.UUID) func()
}

type Locker interface {
	Acquire(ctx context.Context) error
	Release(ctx context.Context) error
}

type NotificationsPollingService struct {
	clock                 clockwork.Clock
	notificationsGetter   NotificationsGetter
	notificationProcessor NotificationsProcessor
	config                Config
	logger                log.Logger
	pollingLock           Locker
	notificationsFilter   NotificationsFilter
}

func NewNotificationsPollingService(
	logger log.Logger,
	clock clockwork.Clock,
	notificationsGetter NotificationsGetter,
	notificationsFilter NotificationsFilter,
	notificationProcessor NotificationsProcessor,
	config Config,
	pollingLock Locker,
) *NotificationsPollingService {
	return &NotificationsPollingService{
		logger:                logger.WithName("PollingService"),
		clock:                 clock,
		notificationsGetter:   notificationsGetter,
		notificationsFilter:   notificationsFilter,
		notificationProcessor: notificationProcessor,
		config:                config,
		pollingLock:           pollingLock,
	}
}

func (p *NotificationsPollingService) Poll(ctx context.Context) {
	ctxWithTimeout, cancelFunc := context.WithTimeout(ctx, p.config.MaxIterationDuration)
	defer cancelFunc()
	err := p.pollingLock.Acquire(ctxWithTimeout)
	if err != nil {
		return
	}
	iterationID, _ := uuid.NewV4()
	p.logger.Info("start polling iteration", log.String("iterationID", iterationID.String()))
	defer func() {
		if err := p.pollingLock.Release(ctx); err != nil {
			p.logger.Error("failed to release polling lock", log.Error(err), log.String("iterationID", iterationID.String()))
		} else {
			p.logger.Info("polling lock has been released", log.String("iterationID", iterationID.String()))
		}
	}()
	notifications, err := p.notificationsGetter.GetFirstN(
		ctx,
		p.clock.Now(),
		p.config.PlannedPerBatch,
		p.config.PostponedPerBatch,
		p.config.NotSentPerBatch,
	)
	if err != nil {
		p.logger.Error(
			"failed to get notifications during polling iteration",
			log.Error(err),
			log.String("iterationID", iterationID.String()),
		)
		return
	}
	notifications = p.applyFilter(notifications, iterationID)

	if len(notifications) == 0 {
		p.logger.Info("0 notifications have been selected on the current iteration", log.String("iterationID", iterationID.String()))
		return
	}
	_ = p.notificationProcessor.Process(ctx, notifications, iterationID)
}

func (p *NotificationsPollingService) applyFilter(notifications []models.Notification, iterationID uuid.UUID) []models.Notification {
	allNotificationsCount := len(notifications)
	notifications = p.notificationsFilter.Filter(iterationID, notifications)
	filteredOutCount := allNotificationsCount - len(notifications)
	if filteredOutCount > 0 {
		p.logger.Info(
			"notifications have been filtered out",
			log.Int("filteredOut", filteredOutCount),
			log.Int("remained", len(notifications)),
			log.String("iterationID", iterationID.String()),
		)
	}
	return notifications
}
