package audioguide

import (
	"context"
	"fmt"
	"net/url"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

const (
	durationStaticImageURL       = "https://yastat.net/s3/travel-mail/icons/Time.png"
	maxNumberOfAudioGuidesToSend = 3
)

type AudioGuidesProvider interface {
	GetAudioGuides(context.Context, int) (structs.AudioGuidesForCity, error)
}

type Provider struct {
	texts                  tanker.Keyset
	routePointsExtractor   interfaces.RoutePointsExtractor
	audioGuidesProvider    AudioGuidesProvider
	settlementDataProvider interfaces.SettlementDataProvider
}

func NewProvider(
	texts tanker.Keyset,
	routePointsExtractor interfaces.RoutePointsExtractor,
	audioGuidesProvider AudioGuidesProvider,
	settlementDataProvider interfaces.SettlementDataProvider,
) *Provider {
	provider := &Provider{
		texts:                  texts,
		routePointsExtractor:   routePointsExtractor,
		audioGuidesProvider:    audioGuidesProvider,
		settlementDataProvider: settlementDataProvider,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	block := ui.NewCarouselBlock()
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return nil, err
	}

	if settlementID <= 0 {
		return block, blocks.ErrUnknownSettlement
	}

	settlementGeoID, found := p.settlementDataProvider.GetGeoID(settlementID)
	if !found {
		return block, fmt.Errorf("settlement doesn't have geoID")
	}

	audioGuides, err := p.audioGuidesProvider.GetAudioGuides(ctx, settlementGeoID)
	if err != nil {
		return block, fmt.Errorf("unable to get audioguides for city: %w", err)
	}
	block.Title = p.texts.GetSingular("title", "ru")
	block.Subtitle = p.texts.GetSingular("subtitle", "ru")

	if audioGuides.DirectURL != "" {
		directURL, err := addCommonQueryParams(audioGuides.DirectURL)
		if err != nil {
			return block, fmt.Errorf("unable to format all.guides URL: %w", err)
		}
		block.Action = ui.SecondaryAction{
			Text:  p.texts.GetSingular("all.guides", "ru"),
			Theme: ui.SecondaryActionTheme,
			URL:   directURL,
		}
	}

	if len(audioGuides.Tours) > 0 {
		for _, tour := range audioGuides.Tours {
			blockItem, shouldBeRendered, err := p.createBlockItem(tour)
			if err != nil {
				return block, err
			}
			if !shouldBeRendered {
				continue
			}
			block.Items = append(block.Items, blockItem)
			if len(block.Items) >= maxNumberOfAudioGuidesToSend {
				break
			}
		}
	}

	if len(block.Items) == 0 {
		return block, fmt.Errorf("no suitable audioguides are found for city: %w", err)
	}

	return block, nil
}

func (p *Provider) createBlockItem(tour structs.AudioGuide) (ui.CarouselBlockItemInterface, bool, error) {
	if tour.TourURL == "" {
		return ui.CommonCarouselBlockItem{}, false, fmt.Errorf("tour's URL is empty")
	}
	tourURL, err := addCommonQueryParams(tour.TourURL)
	if err != nil {
		return ui.CommonCarouselBlockItem{}, false, fmt.Errorf("unable to format tour's URL")
	}
	item := ui.CommonCarouselBlockItem{
		CarouselBlockItemBase: ui.CarouselBlockItemBase{
			Image: tour.ImageURL,
			Title: tour.Name,
			URL:   tourURL,
		},
	}
	itemShouldBeRendered := true
	switch tour.Type {
	case "museum":
		item.Description = p.texts.GetSingular("museum", "ru")
	case "tour":
		switch tour.Category {
		case "walk":
			item.Description = p.texts.GetSingular("walk.tour", "ru")
		case "car":
			item.Description = p.texts.GetSingular("car.tour", "ru")
		default:
			itemShouldBeRendered = false
		}
	default:
		itemShouldBeRendered = false
	}
	if tour.Duration != 0 {
		hours := tour.Duration / 3600
		minutes := (tour.Duration % 3600) / 60
		var durationText string
		if hours > 0 {
			if minutes > 0 {
				durationText = fmt.Sprintf(
					"%d %s %d %s",
					hours,
					p.texts.GetPlural("hour", "ru", hours),
					minutes,
					p.texts.GetPlural("minute", "ru", minutes),
				)
			} else {
				durationText = fmt.Sprintf("%d %s", hours, p.texts.GetPlural("hour", "ru", hours))
			}
		} else {
			durationText = fmt.Sprintf("%d %s", minutes, p.texts.GetPlural("minute", "ru", minutes))
		}

		if durationText != "" {
			item.Info = ui.CarouselIconAndText{
				Icon: durationStaticImageURL,
				Text: durationText,
			}
		}
	}
	return item, itemShouldBeRendered, nil
}

func addCommonQueryParams(originalURL string) (string, error) {
	parsedURL, err := url.Parse(originalURL)
	if err != nil {
		return originalURL, err
	}

	query := parsedURL.Query()
	query.Add("utm_campaign", "YT")
	query.Add("utm_medium", "PreTrip")
	query.Add("utm_source", "YandexTravel")
	parsedURL.RawQuery = query.Encode()
	return parsedURL.String(), nil
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.AudioGuideBlock
}
