package blocks

import (
	"context"
	"errors"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/notifier/internal/collecting"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
)

type BlockTypesSet map[BlockType]bool

type BlockProvider interface {
	GetBlock(context.Context, *orders.OrderInfo, models.Notification) (renderer.Block, error)
	GetBlockType() BlockType
}

type DisclaimersBlockProvider interface {
	GetDisclaimersBlock(BlockTypesSet) (renderer.Block, error)
}

type Collector struct {
	logger                   log.Logger
	blockProviderByType      map[BlockType]BlockProvider
	disclaimersBlockProvider DisclaimersBlockProvider
}

type CollectorOption func(*Collector)

func NewBlocksCollector(logger log.Logger, options ...CollectorOption) *Collector {
	collector := &Collector{
		logger:              logger.WithName("PretripBlocksCollector"),
		blockProviderByType: make(map[BlockType]BlockProvider),
	}

	for _, option := range options {
		option(collector)
	}
	return collector
}

func WithBlockProvider(provider BlockProvider) CollectorOption {
	return func(processor *Collector) {
		processor.blockProviderByType[provider.GetBlockType()] = provider
	}
}

func WithDisclaimersBlockProvider(provider DisclaimersBlockProvider) CollectorOption {
	return func(processor *Collector) {
		processor.disclaimersBlockProvider = provider
	}
}

func (c *Collector) GetBlocks(
	ctx context.Context,
	blockConfigs []BlockConfig,
	orderInfo *orders.OrderInfo,
	notification models.Notification,
) ([]renderer.Block, error) {
	regularBlocksCollector := collecting.NewParallelCollector()
	var disclaimersBlockProvider DisclaimersBlockProvider
	collectedBlockTypes := make(BlockTypesSet)
	collectedBlockTypesMutex := sync.RWMutex{}
	for _, blockConfig := range blockConfigs {
		blockType := blockConfig.Type
		checkSubtype := func(i int) bool {
			return blockConfig.IncludedIn[i] == notification.Subtype
		}
		if !any(len(blockConfig.IncludedIn), checkSubtype) {
			continue
		}
		checkOrderType := func(i int) bool {
			return blockConfig.ExcludedForOrderType[i] == notification.Order.Type
		}
		if any(len(blockConfig.ExcludedForOrderType), checkOrderType) {
			continue
		}
		required := blockConfig.Required
		if blockType == DisclaimersBlock && c.disclaimersBlockProvider != nil {
			disclaimersBlockProvider = c.disclaimersBlockProvider
		} else if blockProvider, ok := c.blockProviderByType[blockType]; ok {
			regularBlocksCollector.AddProvider(
				func() (interface{}, error) {
					block, err := blockProvider.GetBlock(ctx, orderInfo, notification)
					if err == nil {
						collectedBlockTypesMutex.Lock()
						defer collectedBlockTypesMutex.Unlock()
						collectedBlockTypes[blockType] = true
					}
					writeMetric(err == nil, blockType, required)
					return block, err
				},
				func(err error) error {
					ctx := ctxlog.WithFields(ctx, log.String("blockType", blockType.String()))
					if required {
						return errGetRequiredBlock{blockType, err}
					}
					if errors.Is(err, ErrDataNotFound) {
						c.logger.Warn("no data found for block", ctxlog.ContextFields(ctx)...)
					} else {
						c.logger.Warn(
							"failed to get data for unnecessary block type",
							append(ctxlog.ContextFields(ctx), log.Error(err))...,
						)
					}
					return nil
				},
			)
		} else if required {
			return nil, errUnknownBlockType{blockType}
		} else {
			c.logger.Warn(
				"no registered provider for unnecessary block type",
				append(ctxlog.ContextFields(ctx), log.String("blockType", blockType.String()))...,
			)
		}
	}
	regularBlocks, err := regularBlocksCollector.Collect()
	if err != nil {
		return nil, err
	}
	blocks := make([]renderer.Block, 0, len(regularBlocks))
	for _, b := range regularBlocks {
		blocks = append(blocks, b.(renderer.Block))
	}
	if disclaimersBlockProvider != nil {
		disclaimersBlock, err := disclaimersBlockProvider.GetDisclaimersBlock(collectedBlockTypes)
		if err != nil {
			return nil, err
		}
		blocks = append(blocks, disclaimersBlock)
	}
	return blocks, nil
}

func any(len int, f func(i int) bool) bool {
	for i := 0; i < len; i++ {
		if f(i) {
			return true
		}
	}
	return false
}

type errGetRequiredBlock struct {
	blockType BlockType
	error     error
}

func (e errGetRequiredBlock) Error() string {
	return fmt.Errorf("failed to get data for required block type %v: %w", e.blockType, e.error).Error()
}

type errUnknownBlockType struct {
	blockType BlockType
}

func (e errUnknownBlockType) Error() string {
	return fmt.Sprintf("no registered provider for required block type %v", e.blockType)
}
