package blocks

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/travel/notifier/internal/models"
)

func init() {
	for s, blockType := range stringToBlockType {
		blockTypeToString[blockType] = s
	}
}

type BlockConfig struct {
	IncludedIn           []models.NotificationSubtype `yaml:"included_in"`
	ExcludedForOrderType []models.OrderType           `yaml:"excluded_for_order_type"`
	Type                 BlockType
	Required             bool
}

type BlockType int8

// List of block types is copied from here
// https://wiki.yandex-team.ru/travel/pretrip-pisma/#soderzhaniezhdpisma
const (
	HeaderBlock BlockType = 1 + iota
	BannerBlock
	GreetingBlock
	HotelVaucherBlock
	TimelapsBlock
	HotelRouteBlock
	NavigationBlock // maps, subway, public transportation
	RidesBlock      // Taxi, Drive
	WeatherBlock
	AudioGuideBlock
	EventsBlock
	MoviesBlock
	DiskBlock
	BookHotelBlock
	HealthBlock
	CovidBlock
	BeforeDepartureBlock
	FooterBlock
	TestingBlock
	DisclaimersBlock
	HotelOrderBlock
	TrainOrderBlock
)

var stringToBlockType = map[string]BlockType{
	"header":           HeaderBlock,
	"banner":           BannerBlock,
	"greeting":         GreetingBlock,
	"hotel_voucher":    HotelVaucherBlock,
	"time_lapse":       TimelapsBlock,
	"hotel_route":      HotelRouteBlock,
	"navigation":       NavigationBlock,
	"taxi":             RidesBlock,
	"weather":          WeatherBlock,
	"audio_guide":      AudioGuideBlock,
	"events":           EventsBlock,
	"movies":           MoviesBlock,
	"disk":             DiskBlock,
	"book_hotel":       BookHotelBlock,
	"health":           HealthBlock,
	"covid":            CovidBlock,
	"before_departure": BeforeDepartureBlock,
	"footer":           FooterBlock,
	"testing":          TestingBlock,
	"disclaimers":      DisclaimersBlock,
	"hotel_order":      HotelOrderBlock,
	"train_order":      TrainOrderBlock,
}

var blockTypeToString = make(map[BlockType]string)

func (t BlockType) String() string {
	return blockTypeToString[t]
}

func (t *BlockType) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var o string
	err := unmarshal(&o)
	if err != nil {
		return err
	}

	if op, ok := stringToBlockType[strings.ToLower(o)]; !ok {
		return fmt.Errorf("could not convert %s to block type", o)
	} else {
		*t = op
	}

	return nil
}
