package events

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/travel/notifier/internal/externalhttp"
	"a.yandex-team.ru/travel/notifier/internal/geo"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

const (
	maxNumberOfEventsToFetch          = 50
	numberOfDaysForwardToFetchEvents  = 2
	maxDistanceFromDestinationAllowed = 100.
)

type AfishaClient struct {
	httpClient *externalhttp.HTTPClient
}

func NewAfishaEventsClient(httpClient *externalhttp.HTTPClient) *AfishaClient {
	return &AfishaClient{
		httpClient: httpClient,
	}
}

func (c *AfishaClient) GetEvents(
	ctx context.Context,
	geoID int,
	startDate time.Time,
	latitude float64,
	longitude float64,
) (structs.EventsForCity, error) {
	result := structs.NewEventsForCity()
	params := map[string]string{
		"resultsLimit": strconv.Itoa(maxNumberOfEventsToFetch),
		"geoId":        strconv.Itoa(geoID),
		"startDate":    startDate.Format("2006-01-02"),
		"period":       strconv.Itoa(numberOfDaysForwardToFetchEvents),
	}
	jsonBytes, err := c.httpClient.GetJSON(ctx, params)
	if err != nil {
		if errors.As(err, &externalhttp.TravelAPINotFoundError{}) {
			return result, blocks.ErrDataNotFound
		}
		return result, err
	}

	if err = json.Unmarshal(jsonBytes, &result); err != nil {
		return result, err
	}

	distance := geo.GetDistanceKM(latitude, longitude, result.Coordinates.Latitude, result.Coordinates.Longitude)
	if distance > maxDistanceFromDestinationAllowed {
		return result, fmt.Errorf("events are too far from the destination")
	}

	return result, nil
}
