package events

import (
	"context"
	"fmt"
	"math"
	"net/url"
	"regexp"
	"strings"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

const (
	eventDateStaticImageURL = "https://yastat.net/s3/travel-mail/icons/Calendar.png"
	maxNumberOfEventsToSend = 3
	maxEventTags            = 3
)

var hasCyrillicLetterOrPlus = regexp.MustCompile(`[абвгдеёжзийклмнопрстуфхцчшщъыьэюя+АБВГДЕЁЖЗИКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ]`)

type Provider struct {
	texts                  tanker.Keyset
	routePointsExtractor   interfaces.RoutePointsExtractor
	eventsProvider         interfaces.EventsProvider
	settlementDataProvider interfaces.SettlementDataProvider
}

func NewProvider(
	texts tanker.Keyset,
	routePointsExtractor interfaces.RoutePointsExtractor,
	eventsProvider interfaces.EventsProvider,
	settlementDataProvider interfaces.SettlementDataProvider,
) *Provider {
	provider := &Provider{
		texts:                  texts,
		routePointsExtractor:   routePointsExtractor,
		eventsProvider:         eventsProvider,
		settlementDataProvider: settlementDataProvider,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	block := ui.NewCarouselBlock()
	order, err := orderInfo.ToOrder()
	if err != nil {
		return block, err
	}
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return block, err
	}
	if settlementID <= 0 {
		return block, blocks.ErrUnknownSettlement
	}
	block.Title = p.texts.GetSingular("title", "ru")

	settlementTitleTranslation, found := p.settlementDataProvider.GetTitleTranslation(settlementID)
	if !found {
		return block, fmt.Errorf("settlement title not found")
	}
	if settlementTitleTranslation.Genitive != "" {
		textParams := make(map[string]interface{})
		textParams["settlementTitle"] = settlementTitleTranslation.Genitive
		subtitle, err := tanker.TemplateToString("subtitle", p.texts.GetSingular("subtitle", "ru"), textParams)
		if err == nil {
			block.Subtitle = subtitle
		}
	}
	if block.Subtitle == "" {
		block.Subtitle = p.texts.GetSingular("subtitle.simple", "ru")
	}

	if order.ArrivalDate.IsZero() {
		return block, fmt.Errorf("unable to get arrival date for the order")
	}
	tz := p.settlementDataProvider.GetSettlementTimeZone(settlementID)
	if tz == nil {
		return block, fmt.Errorf("unable to get destination timezone for the order")
	}

	settlementGeoID, found := p.settlementDataProvider.GetGeoID(settlementID)
	if !found {
		return block, fmt.Errorf("settlement doesn't have geoID")
	}

	latitude, longitude, found := p.settlementDataProvider.GetCoordinates(settlementID)
	if !found {
		return block, fmt.Errorf("settlement doesn't have coordinates")
	}

	events, err := p.eventsProvider.GetEvents(ctx, settlementGeoID, order.ArrivalDate.In(tz), latitude, longitude)
	if err != nil || len(events.Events) == 0 {
		return block, fmt.Errorf("unable to get events for city: %w", err)
	}

	events.Events = filterEvents(
		events.Events,
		func(event structs.Event) bool {
			return event.Name != "" && event.ImageURL != "" && event.EventURL != ""
		},
		maxNumberOfEventsToSend,
	)
	if len(events.Events) == 0 {
		return block, fmt.Errorf("all fetched events are missing some vital info")
	}

	if events.RegionURL != "" {
		regionURLWithUTMs, err := addCommonQueryParams(events.RegionURL)
		if err != nil {
			return block, fmt.Errorf("unable to format events region URL")
		}
		block.Action = ui.SecondaryAction{
			Text:  p.texts.GetSingular("region.events", "ru"),
			Theme: ui.SecondaryActionTheme,
			URL:   regionURLWithUTMs,
		}
	}

	if len(events.Events) > 0 {
		for _, event := range events.Events {
			if event.EventURL != "" {
				eventURLWithUTMs, err := addCommonQueryParams(event.EventURL)
				if err != nil {
					return block, fmt.Errorf("unable to format event's URL")
				}
				event.EventURL = eventURLWithUTMs
			}
			block.Items = append(block.Items, p.createBlockItem(event))
		}
	}

	return block, nil
}

func (p *Provider) createBlockItem(event structs.Event) ui.CarouselBlockItemInterface {
	item := ui.CommonCarouselBlockItem{
		CarouselBlockItemBase: ui.CarouselBlockItemBase{
			Image: event.ImageURL,
			Title: event.Name,
			URL:   event.EventURL,
		},
	}

	descriptionParts := []string{}
	minPriceAmount, minPriceCharacter, ok := p.formatMinPrice(event.MinPrice)
	if ok {
		textParams := make(map[string]interface{})
		textParams["minPrice"] = minPriceAmount
		textParams["currency"] = minPriceCharacter
		pricePart, err := tanker.TemplateToString("price.part", p.texts.GetSingular("price.part", "ru"), textParams)
		if err == nil {
			descriptionParts = append(descriptionParts, pricePart)
		}
	}
	if event.Type != "" {
		descriptionParts = append(descriptionParts, event.Type)
	}

	if event.Tags != nil {
		eventTagsCount := 0
		for _, tag := range event.Tags {
			if eventTagsCount >= maxEventTags {
				break
			}
			if isAcceptableTag(tag, descriptionParts) {
				descriptionParts = append(descriptionParts, tag)
				eventTagsCount++
			}
		}
	}
	item.Description = strings.Join(descriptionParts, constants.MiddleDotSeparator)

	if event.DateText != "" {
		item.Info = ui.CarouselIconAndText{
			Icon: eventDateStaticImageURL,
			Text: event.DateText,
		}
	}
	return item
}

func addCommonQueryParams(originalURL string) (string, error) {
	parsedURL, err := url.Parse(originalURL)
	if err != nil {
		return originalURL, err
	}

	query := parsedURL.Query()
	query.Add("utm_campaign", "email")
	query.Add("utm_medium", "PreTrip")
	query.Add("utm_source", "yandex-travel")
	parsedURL.RawQuery = query.Encode()
	return parsedURL.String(), nil
}

func isAcceptableTag(tag string, alreadyAcceptedTags []string) bool {
	for _, existing := range alreadyAcceptedTags {
		if tag == existing {
			return false
		}
	}
	return hasCyrillicLetterOrPlus.MatchString(tag)
}

func (p *Provider) formatMinPrice(price *structs.Price) (minPriceAmount, minPriceCharacter string, ok bool) {
	if price == nil || price.Value <= 0 || price.Currency == "" {
		return
	}
	minPriceCharacter, ok = constants.CurrencyCharactersMap[price.Currency]
	if !ok {
		return
	}
	minPriceAmount = fmt.Sprint(math.Round(price.Value))
	return
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.EventsBlock
}

func filterEvents(events []structs.Event, predicate func(structs.Event) bool, maxCount int) []structs.Event {
	result := make([]structs.Event, 0, maxCount)
	for _, event := range events {
		if !predicate(event) {
			continue
		}
		result = append(result, event)
		if len(result) >= maxCount {
			break
		}
	}
	return result
}
