package greeting

import (
	"context"
	"fmt"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
)

const (
	luggageStaticImageURL = "https://yastat.net/s3/travel-mail/images/luggage.png"
)

type Provider struct {
	texts                  tanker.Keyset
	routePointsExtractor   interfaces.RoutePointsExtractor
	cityImagesProvider     interfaces.CityImagesProvider
	settlementDataProvider interfaces.SettlementDataProvider
}

func NewProvider(
	texts tanker.Keyset,
	routePointsExtractor interfaces.RoutePointsExtractor,
	cityImagesProvider interfaces.CityImagesProvider,
	settlementDataProvider interfaces.SettlementDataProvider,
) *Provider {
	provider := &Provider{
		texts:                  texts,
		routePointsExtractor:   routePointsExtractor,
		cityImagesProvider:     cityImagesProvider,
		settlementDataProvider: settlementDataProvider,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	block := ui.NewGreeting("", "", "", false)
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return nil, err
	}
	if settlementID <= 0 {
		return block, blocks.ErrUnknownSettlement
	}
	settlementTitleTranslation, found := p.settlementDataProvider.GetTitleTranslation(settlementID)
	if !found {
		return block, blocks.ErrNoSettlementTitleTranslation
	}
	textParams := make(map[string]interface{})
	textParams["settlementTitle"] = settlementTitleTranslation.Nominative
	if accusativeWithPrep, ok := p.settlementDataProvider.GetAccusativeTitleWithPreposition(settlementID); ok {
		textParams["settlementAccusativeWithPrep"] = accusativeWithPrep
	}
	textParams["settlementPreposition"] = p.settlementDataProvider.GetPreposition(settlementID)
	textParams["cityLoc"] = settlementTitleTranslation.Prepositional

	var title string
	var text string
	var errText, errTitle error
	switch notification.Subtype {
	case models.NotificationWeekBefore:
		text, errText = tanker.TemplateToString("week.text", p.texts.GetSingular("week.text", "ru"), textParams)
		title, errTitle = tanker.TemplateToString("week.title", p.texts.GetSingular("week.title", "ru"), textParams)
	case models.NotificationDayBefore:
		text, errText = tanker.TemplateToString("day.text", p.texts.GetSingular("day.text", "ru"), textParams)
		title, errTitle = tanker.TemplateToString("default.title", p.texts.GetSingular("default.title", "ru"), textParams)
		block.Image = luggageStaticImageURL
		block.Overlaying = true
	case models.NotificationAdhoc:
		text, errText = tanker.TemplateToString("adhoc.text", p.texts.GetSingular("adhoc.text", "ru"), textParams)
		title, errTitle = tanker.TemplateToString("default.title", p.texts.GetSingular("default.title", "ru"), textParams)
	default:
		return block, fmt.Errorf("unknown notification subtype")
	}
	if errText != nil {
		return block, errText
	}
	block.Text = text

	if errTitle != nil {
		return block, errTitle
	}
	block.Title = title

	if notification.Subtype != models.NotificationDayBefore {
		image, err := p.cityImagesProvider.GetCityImageByID(ctx, settlementID)
		if err != nil {
			return block, err
		}
		if image == nil {
			return block, fmt.Errorf("no image for block from the content admin")
		}
		block.Image = image.ImageUrl
	}

	return block, nil
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.GreetingBlock
}
