package hotelorder

import (
	"context"
	"fmt"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/travelapi"
)

const (
	anyTimeCheckInOutValue = "в любое время"
	defaultHotelImageSize  = "L"
)

type Provider struct {
	texts                       tanker.Keyset
	additionalOrderInfoProvider interfaces.AdditionalOrderInfoProvider
}

func NewProvider(
	texts tanker.Keyset,
	additionalOrderInfoProvider interfaces.AdditionalOrderInfoProvider,
) *Provider {
	return &Provider{
		texts:                       texts,
		additionalOrderInfoProvider: additionalOrderInfoProvider,
	}
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	additionalInfo, err := p.additionalOrderInfoProvider.GetAdditionalOrderInfo(ctx, orderInfo.ID)
	if err != nil {
		return nil, fmt.Errorf("unable to get additional info for orderID %s: %w", orderInfo.ID, err)
	}
	if additionalInfo == nil || len(additionalInfo.HotelOrderInfos) == 0 {
		return nil, fmt.Errorf("no additional info for orderID %s: %v", orderInfo.ID, additionalInfo)
	}
	hotelOrderInfo := additionalInfo.HotelOrderInfos[0]
	title := p.texts.GetSingular("title", "ru")
	hotelGeoPosition := ui.Link{URL: "#", Text: hotelOrderInfo.Address}
	dates, err := p.buildDates(hotelOrderInfo)
	if err != nil {
		return nil, fmt.Errorf(
			"failed to render check-in/check-out dates for orderID %s: %s – %s",
			orderInfo.ID,
			hotelOrderInfo.CheckInDate,
			hotelOrderInfo.CheckOutDate,
		)
	}
	checkInTimeInterval, err := p.buildCheckIn(hotelOrderInfo.CheckInBeginTime, hotelOrderInfo.CheckInEndTime)
	if err != nil {
		return nil, fmt.Errorf(
			"failed to render check-in time interval for orderID %s: %s – %s",
			orderInfo.ID,
			hotelOrderInfo.CheckInBeginTime,
			hotelOrderInfo.CheckInEndTime,
		)
	}
	checkOutUpperBound, err := p.buildCheckOut(hotelOrderInfo.CheckOutTime, checkInTimeInterval != "")
	if err != nil {
		return nil, fmt.Errorf(
			"failed to render check-out time upper bound for orderID %s: %s", orderInfo.ID, hotelOrderInfo.CheckOutTime,
		)
	}

	return ui.NewHotelOrder(
		title,
		hotelOrderInfo.Name,
		hotelGeoPosition,
		dates,
		checkInTimeInterval,
		checkOutUpperBound,
		p.buildDownload(hotelOrderInfo),
		p.buildImageURL(hotelOrderInfo.ImageURLTemplate),
	), nil
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.HotelOrderBlock
}

func (p *Provider) buildDates(hotelOrderInfo travelapi.HotelOrderInfo) (string, error) {
	checkInDate, err := time.Parse(timeformats.RFC3339Date, hotelOrderInfo.CheckInDate)
	if err != nil {
		return "", err
	}
	checkOutDate, err := time.Parse(timeformats.RFC3339Date, hotelOrderInfo.CheckOutDate)
	if err != nil {
		return "", err
	}
	textParams := map[string]interface{}{
		"startDay":     checkInDate.Day(),
		"startMonth":   constants.GenitiveMonthNames[checkInDate.Month()],
		"startDayName": constants.WeekdayShortNames[checkInDate.Weekday()],
		"endDay":       checkOutDate.Day(),
		"endMonth":     constants.GenitiveMonthNames[checkOutDate.Month()],
		"endDayName":   constants.WeekdayShortNames[checkOutDate.Weekday()],
	}
	return tanker.TemplateToString("dates", p.texts.GetSingular("dates", "ru"), textParams)
}

func (p *Provider) buildCheckIn(beginTime, endTime string) (string, error) {
	if strings.ToLower(beginTime) != anyTimeCheckInOutValue && strings.ToLower(endTime) == anyTimeCheckInOutValue {
		textParams := map[string]interface{}{"beginTime": beginTime}
		return tanker.TemplateToString("checkin", p.texts.GetSingular("checkin", "ru"), textParams)
	}
	if strings.ToLower(beginTime) == anyTimeCheckInOutValue || strings.ToLower(endTime) == anyTimeCheckInOutValue {
		return p.texts.GetSingular("checkin_any_time", "ru"), nil
	}
	if beginTime == "" && endTime == "" {
		return "", nil
	}
	textParams := map[string]interface{}{
		"beginTime": beginTime,
		"endTime":   endTime,
	}
	return tanker.TemplateToString("checkin", p.texts.GetSingular("checkin", "ru"), textParams)
}

func (p *Provider) buildCheckOut(checkOutEndTime string, hasCheckInInterval bool) (checkOutInterval string, err error) {
	if strings.ToLower(checkOutEndTime) == anyTimeCheckInOutValue {
		checkOutInterval = p.texts.GetSingular("checkout_any_time", "ru")
	} else {
		textParams := map[string]interface{}{"endTime": checkOutEndTime}
		checkOutInterval, err = tanker.TemplateToString("checkout", p.texts.GetSingular("checkout", "ru"), textParams)
		if err != nil {
			return "", err
		}
	}
	if hasCheckInInterval {
		return strings.ToLower(checkOutInterval), nil
	}
	return checkOutInterval, nil
}

func (p *Provider) buildDownload(hotelOrderInfo travelapi.HotelOrderInfo) *ui.SecondaryAction {
	if hotelOrderInfo.DocumentURL == "" {
		return nil
	}
	return &ui.SecondaryAction{
		Text:  p.texts.GetSingular("download_voucher", "ru"),
		Theme: ui.SecondaryActionTheme,
		URL:   hotelOrderInfo.DocumentURL,
	}
}

func (p *Provider) buildImageURL(template string) string {
	return fmt.Sprintf(template, defaultHotelImageSize)
}
