package hotels

import (
	"context"
	"fmt"
	"math"
	"net/url"
	"sort"
	"strconv"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

type Provider struct {
	texts                  tanker.Keyset
	routePointsExtractor   interfaces.RoutePointsExtractor
	hotelsProvider         interfaces.HotelsProvider
	settlementDataProvider interfaces.SettlementDataProvider
	travelPortalURL        string
}

const (
	hotelsLimit = 3
)

var (
	errInvalidPrice    = fmt.Errorf("invalid price")
	errUnknownCurrency = fmt.Errorf("unknown currency")
)

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.BookHotelBlock
}

func NewProvider(
	texts tanker.Keyset,
	routePointsExtractor interfaces.RoutePointsExtractor,
	hotelsProvider interfaces.HotelsProvider,
	settlementDataProvider interfaces.SettlementDataProvider,
	travelPortalURL string,
) *Provider {
	provider := &Provider{
		texts:                  texts,
		routePointsExtractor:   routePointsExtractor,
		hotelsProvider:         hotelsProvider,
		settlementDataProvider: settlementDataProvider,
		travelPortalURL:        travelPortalURL,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	settlementID, settlementGeoID, err := p.getSettlementIDs(orderInfo)
	if err != nil {
		return nil, err
	}
	block := ui.NewCarouselBlock()
	block.Title = p.texts.GetSingular("title", "ru")

	settlementTitleTranslation, found := p.settlementDataProvider.GetTitleTranslation(settlementID)
	if !found {
		return nil, blocks.ErrNoSettlementTitleTranslation
	}
	if settlementTitleTranslation.Genitive != "" {
		textParams := make(map[string]interface{})
		textParams["settlementTitle"] = settlementTitleTranslation.Genitive
		subtitle, err := tanker.TemplateToString("subtitle", p.texts.GetSingular("subtitle", "ru"), textParams)
		if err == nil {
			block.Subtitle = subtitle
		}
	}
	if block.Subtitle == "" {
		block.Subtitle = p.texts.GetSingular("subtitle.simple", "ru")
	}

	hotelPayload, err := p.hotelsProvider.GetHotels(ctx, settlementGeoID, orderInfo.ID, hotelsLimit)
	if err != nil {
		return nil, fmt.Errorf("unable to get hotels for settlementGeoID %d: %w", settlementGeoID, err)
	}
	hotelItems := hotelPayload.HotelsList
	if len(hotelItems) == 0 {
		return nil, fmt.Errorf("no hotels were found for settlementGeoID: %d", settlementGeoID)
	}

	for _, hotelItem := range hotelItems {
		blockItem, err := p.createBlockItem(hotelPayload, hotelItem)
		if err != nil {
			continue
		}
		block.Items = append(block.Items, blockItem)
	}

	block.Action = ui.SecondaryAction{
		Theme: ui.SecondaryActionTheme,
		URL:   p.buildAllHotelsURL(hotelPayload),
		Text:  p.texts.GetSingular("all.hotels", "ru"),
	}

	return block, nil
}

func (p *Provider) createBlockItem(hotelPayload *structs.HotelPayload, hotelItem structs.HotelsBlockItem) (
	ui.CarouselBlockItemInterface,
	error,
) {
	if len(hotelItem.Hotel.Images) == 0 {
		return nil, fmt.Errorf("no images for hotel %s", hotelItem.Hotel.Name)
	}

	price, err := p.formatPrice(hotelItem.MinPrice)
	if err != nil {
		return nil, err
	}

	image := hotelItem.Hotel.Images[0]
	sortImageSizes(image.Sizes)
	imageSize := image.Sizes[0].Size
	orderInfo := make([]string, 0)
	for _, amenity := range hotelItem.Hotel.MainAmenities {
		orderInfo = append(orderInfo, amenity.Name)
	}
	return &ui.HotelCarouselBlockItem{
		CarouselBlockItemBase: ui.CarouselBlockItemBase{
			Image: fmt.Sprintf(image.URLTemplate, imageSize),
			Title: hotelItem.Hotel.Name,
			URL:   p.buildLandingURL(hotelItem.LandingURL, hotelPayload),
		},
		Rating:        hotelItem.Hotel.Rating,
		Reviews:       p.formatReviews(hotelItem.Hotel.TotalTextReviewCount),
		Stars:         hotelItem.Hotel.Stars,
		OrderInfo:     orderInfo,
		Price:         price,
		Accommodation: hotelItem.Hotel.Category.Name,
	}, nil
}

func (p *Provider) getSettlementIDs(orderInfo *orders.OrderInfo) (int, int, error) {
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return 0, 0, err
	}
	if settlementID <= 0 {
		return 0, 0, blocks.ErrUnknownSettlement
	}
	settlementGeoID, found := p.settlementDataProvider.GetGeoID(settlementID)
	if !found {
		return 0, 0, fmt.Errorf("settlement doesn't have geoID")
	}
	return settlementID, settlementGeoID, nil
}

func (p *Provider) formatPrice(price *structs.HotelPrice) (string, error) {
	if price == nil || price.Value <= 0 || price.Currency == "" {
		return "", errInvalidPrice
	}
	currencyCharacter, ok := constants.CurrencyCharactersMap[price.Currency]
	if !ok {
		return "", errUnknownCurrency
	}
	priceValue := fmt.Sprint(math.Round(price.Value))

	params := make(map[string]interface{})
	params["price"] = priceValue
	params["currency"] = currencyCharacter
	priceText, err := tanker.TemplateToString("price", p.texts.GetSingular("price", "ru"), params)
	if err != nil {
		return "", err
	}
	return priceText, nil
}

func (p *Provider) formatReviews(reviewsCount int) string {
	return fmt.Sprintf("%d %s", reviewsCount, p.texts.GetPlural("reviews", "ru", reviewsCount))
}

func (p *Provider) buildAllHotelsURL(hotelPayload *structs.HotelPayload) string {
	urlPath := fmt.Sprintf("%s/hotels/search", p.travelPortalURL)
	parsedURL, _ := url.Parse(urlPath)
	query := parsedURL.Query()
	query.Add("geoId", strconv.Itoa(hotelPayload.RegionGeoID))
	p.addCommonQueryParams(query, hotelPayload)
	parsedURL.RawQuery = query.Encode()
	return parsedURL.String()
}

func (p *Provider) buildLandingURL(landingURL string, hotelPayload *structs.HotelPayload) string {
	urlPath := fmt.Sprintf("%s/hotels/%s", p.travelPortalURL, landingURL)
	parsedURL, _ := url.Parse(urlPath)
	query := parsedURL.Query()
	p.addCommonQueryParams(query, hotelPayload)
	parsedURL.RawQuery = query.Encode()
	return parsedURL.String()
}

func (p *Provider) addCommonQueryParams(query url.Values, hotelPayload *structs.HotelPayload) {
	query.Add("adults", strconv.Itoa(hotelPayload.RequestParams.Adults))
	childrenAges := make([]string, 0, len(hotelPayload.RequestParams.ChildrenAges))
	for _, age := range hotelPayload.RequestParams.ChildrenAges {
		childrenAges = append(childrenAges, strconv.Itoa(age))
	}
	query.Add("childrenAges", strings.Join(childrenAges, ","))
	query.Add("checkinDate", time.Time(hotelPayload.RequestParams.CheckinDate).Format(timeformats.RFC3339Date))
	query.Add("checkoutDate", time.Time(hotelPayload.RequestParams.CheckoutDate).Format(timeformats.RFC3339Date))
	query.Add("utm_campaign", "email")
	query.Add("utm_medium", "PreTrip")
	query.Add("utm_source", "carousel")
}

var imageSizesOrder = map[string]int{
	"XXXS": 1,
	"XXS":  2,
	"XS":   3,
	"S":    4,
	"M":    5,
	"L":    6,
	"XL":   7,
	"XXL":  8,
	"XXXL": 9,
	"orig": 10,
}

func sortImageSizes(imageSizes []structs.HotelImageSizes) {
	sort.SliceStable(
		imageSizes, func(i, j int) bool {
			iOrder, ok := imageSizesOrder[imageSizes[i].Size]
			if !ok {
				iOrder = 0
			}
			jOrder, ok := imageSizesOrder[imageSizes[j].Size]
			if !ok {
				jOrder = 0
			}
			return iOrder > jOrder
		},
	)
}
