package navigation

import (
	"context"

	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
)

const utmMarker = "utm_campaign=email&utm_medium=PreTrip&utm_source=icon"

type Provider struct {
	texts                 tanker.Keyset
	routePointsExtractor  interfaces.RoutePointsExtractor
	settlementsWithSubway containers.Set[int]
}

type Option func(*Provider)

func WithSubwaySettlements(settlementIDs []int) Option {
	return func(provider *Provider) {
		provider.settlementsWithSubway = containers.SetOf(settlementIDs...)
	}
}

func NewProvider(texts tanker.Keyset, routePointsExtractor interfaces.RoutePointsExtractor, opts ...Option) *Provider {
	provider := &Provider{
		texts:                 texts,
		routePointsExtractor:  routePointsExtractor,
		settlementsWithSubway: defaultSettlementsWithSubway,
	}
	for _, opt := range opts {
		opt(provider)
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return nil, err
	}
	textParams := make(map[string]interface{})
	textParams["utmMarker"] = utmMarker

	block := ui.NewUsefulBlock(
		p.texts.GetSingular("block.title", "ru"),
		"",
		make([]ui.UsefulBlockItem, 0),
	)
	if p.settlementsWithSubway.Contains(settlementID) {
		url, err := tanker.TemplateToString("subway.url", p.texts.GetSingular("subway.url", "ru"), textParams)
		if err != nil {
			return block, err
		}
		block.Items = append(
			block.Items, ui.UsefulBlockItem{
				Image:       p.texts.GetSingular("subway.image", "ru"),
				Title:       p.texts.GetSingular("subway.title", "ru"),
				Description: p.texts.GetSingular("subway.description", "ru"),
				URL:         url,
			},
		)
	}
	url, err := tanker.TemplateToString("commuter.url", p.texts.GetSingular("commuter.url", "ru"), textParams)
	if err != nil {
		return block, err
	}
	block.Items = append(
		block.Items, ui.UsefulBlockItem{
			Image:       p.texts.GetSingular("commuter.image", "ru"),
			Title:       p.texts.GetSingular("commuter.title", "ru"),
			Description: p.texts.GetSingular("commuter.description", "ru"),
			URL:         url,
		},
	)
	return block, nil
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.NavigationBlock
}
