package rides

import (
	"context"

	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
)

type Provider struct {
	texts                tanker.Keyset
	routePointsExtractor interfaces.RoutePointsExtractor
	settlementsWithTaxi  containers.Set[int]
	settlementsWithDrive containers.Set[int]
}

type Option func(*Provider)

func WithTaxiSettlements(settlementIDs []int) Option {
	return func(provider *Provider) {
		provider.settlementsWithTaxi = containers.SetOf(settlementIDs...)
	}
}

func WithDriveSettlements(settlementIDs []int) Option {
	return func(provider *Provider) {
		provider.settlementsWithDrive = containers.SetOf(settlementIDs...)
	}
}

func NewProvider(texts tanker.Keyset, routePointsExtractor interfaces.RoutePointsExtractor, opts ...Option) *Provider {
	provider := &Provider{
		texts:                texts,
		routePointsExtractor: routePointsExtractor,
		settlementsWithTaxi:  defaultSettlementsWithTaxi,
		settlementsWithDrive: defaultSettlementsWithDrive,
	}
	for _, opt := range opts {
		opt(provider)
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return nil, err
	}
	if !p.settlementsWithTaxi.Contains(settlementID) && !p.settlementsWithDrive.Contains(settlementID) {
		return nil, nil
	}
	block := ui.NewUsefulBlock(
		p.texts.GetSingular("block.title", "ru"),
		p.texts.GetSingular("block.subtitle", "ru"),
		make([]ui.UsefulBlockItem, 0),
	)
	if p.settlementsWithTaxi.Contains(settlementID) {
		block.Items = append(
			block.Items, ui.UsefulBlockItem{
				Image:       p.texts.GetSingular("taxi.image", "ru"),
				Title:       p.texts.GetSingular("taxi.title", "ru"),
				Description: p.texts.GetSingular("taxi.description", "ru"),
				URL:         p.texts.GetSingular("taxi.url", "ru"),
			},
		)
	}
	if p.settlementsWithDrive.Contains(settlementID) {
		block.Items = append(
			block.Items, ui.UsefulBlockItem{
				Image:       p.texts.GetSingular("drive.image", "ru"),
				Title:       p.texts.GetSingular("drive.title", "ru"),
				Description: p.texts.GetSingular("drive.description", "ru"),
				URL:         p.texts.GetSingular("drive.url", "ru"),
			},
		)
	}
	return block, nil
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.RidesBlock
}
