package trainorder

import (
	"context"
	"fmt"
	"math"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/strutil"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/travelapi"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type Provider struct {
	texts                       tanker.Keyset
	additionalOrderInfoProvider interfaces.AdditionalOrderInfoProvider
	stationDataProvider         interfaces.StationDataProvider
	timeZoneDataProvider        interfaces.TimeZoneDataProvider
	travelPortalURL             string
}

func NewProvider(
	texts tanker.Keyset,
	stationDataProvider interfaces.StationDataProvider,
	timeZoneDataProvider interfaces.TimeZoneDataProvider,
	additionalOrderInfoProvider interfaces.AdditionalOrderInfoProvider,
	travelPortalURL string,
) *Provider {
	provider := &Provider{
		texts:                       texts,
		stationDataProvider:         stationDataProvider,
		timeZoneDataProvider:        timeZoneDataProvider,
		additionalOrderInfoProvider: additionalOrderInfoProvider,
		travelPortalURL:             travelPortalURL,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	additionalInfo, err := p.additionalOrderInfoProvider.GetAdditionalOrderInfo(ctx, orderInfo.ID)
	if err != nil {
		return nil, fmt.Errorf("unable to get additional info for orderID %s: %w", orderInfo.ID, err)
	}
	if additionalInfo == nil || len(additionalInfo.TrainOrderInfos) == 0 {
		return nil, fmt.Errorf("no additional train infos for orderID %s: %v", orderInfo.ID, additionalInfo)
	}
	trainOrderInfo := additionalInfo.TrainOrderInfos[0]

	departureStation, departureTz, err := p.getStationAndTimeZone(trainOrderInfo.StationFromID)
	if err != nil {
		return nil, fmt.Errorf("unable to parse departure info for order %s: %s", orderInfo.ID, err.Error())
	}

	arrivalStation, arrivalTz, err := p.getStationAndTimeZone(trainOrderInfo.StationToID)
	if err != nil {
		return nil, fmt.Errorf("unable to parse arrival info for order %s: %s", orderInfo.ID, err.Error())
	}

	block := ui.NewTrainOrderBlock()
	block.Title = p.texts.GetSingular("title", "ru")

	departureTime, err := getTime(trainOrderInfo.Departure, departureTz)
	if err != nil {
		return nil, fmt.Errorf("invalid train departure for orderID %s: %v", orderInfo.ID, trainOrderInfo.Departure)
	}
	arrivalTime, err := getTime(trainOrderInfo.Arrival, arrivalTz)
	if err != nil {
		return nil, fmt.Errorf("invalid train arrival for orderID %s: %v", orderInfo.ID, trainOrderInfo.Arrival)
	}
	duration := int(arrivalTime.Sub(departureTime).Minutes())
	durationDays := duration / 1440

	textParams := map[string]interface{}{
		"prettyOrderID":   additionalInfo.PrettyID,
		"durationDays":    strconv.Itoa(durationDays),
		"durationHours":   strconv.Itoa((duration % 1440) / 60),
		"durationMinutes": strconv.Itoa(duration % 60),
		"carNumber":       trainOrderInfo.GetCarNumber(),
	}

	block.OrderNumber, err = tanker.TemplateToString("order.number", p.texts.GetSingular("order.number", "ru"), textParams)
	if err != nil {
		return nil, fmt.Errorf("invalid order number text for orderID %s: %v", orderInfo.ID, err.Error())
	}
	points, err := p.getBlockPoints(departureStation, arrivalStation)
	if err != nil {
		return nil, fmt.Errorf("invalid end point for orderID %s: %v", orderInfo.ID, err.Error())
	}
	block.Points = points
	block.Download = &ui.SecondaryAction{
		Text:  p.texts.GetSingular("download", "ru"),
		Theme: ui.SecondaryActionTheme,
		URL:   fmt.Sprintf("%s/download/trains/ticket/%v", p.travelPortalURL, orderInfo.ID),
	}
	block.Duration, err = p.getDuration(durationDays, textParams)
	if err != nil {
		return nil, err
	}

	block.TrainName = getTrainName(trainOrderInfo)
	block.TrainInfo, err = p.getCarInfoText(trainOrderInfo, textParams)
	if err != nil {
		return nil, err
	}

	departure, err := p.getTrainStationInfo(departureStation, departureTime, departureTime)
	if err != nil {
		return nil, fmt.Errorf("invalid departure station for orderID %s: %v, error=%v", orderInfo.ID, trainOrderInfo, err.Error())
	}
	block.Departure = departure

	arrival, err := p.getTrainStationInfo(arrivalStation, arrivalTime, departureTime)
	if err != nil {
		return nil, fmt.Errorf("invalid arrival station for orderID %s: %v, error=%v", orderInfo.ID, trainOrderInfo, err.Error())
	}
	block.Arrival = arrival

	return block, nil
}

func (p *Provider) getDuration(durationDays int, textParams map[string]interface{}) (string, error) {
	var err error
	var duration string
	if durationDays >= 1 {
		duration, err = tanker.TemplateToString("duration.days", p.texts.GetSingular("duration.days", "ru"), textParams)
		if err != nil {
			return "", fmt.Errorf("invalid duration.days text: %v", err.Error())
		}
	} else {
		duration, err = tanker.TemplateToString("duration.hours", p.texts.GetSingular("duration.hours", "ru"), textParams)
		if err != nil {
			return "", fmt.Errorf("invalid duration.hours text: %v", err.Error())
		}
	}
	return duration, nil
}

func (p *Provider) getCarInfoText(trainOrderInfo travelapi.TrainOrderInfo, textParams map[string]interface{}) (string, error) {
	var err error
	var carInfo string
	if trainOrderInfo.GetCarNumber() != "" {
		carInfo, err = tanker.TemplateToString("car.info", p.texts.GetSingular("car.info", "ru"), textParams)
		if err != nil {
			return "", fmt.Errorf("invalid car info text: %v", err.Error())
		}
	}
	carType := p.texts.GetSingular(fmt.Sprintf("cartype.%s", trainOrderInfo.CarType), "ru")
	var carPlaceNumbers string
	if trainOrderInfo.PlaceNumbers != nil && len(trainOrderInfo.PlaceNumbers) > 0 {
		params := map[string]interface{}{
			"placeNumbers": strings.Join(trainOrderInfo.PlaceNumbers, ", "),
		}
		if len(trainOrderInfo.PlaceNumbers) == 1 {
			carPlaceNumbers, err = tanker.TemplateToString("place.number", p.texts.GetSingular("place.number", "ru"), params)
			if err != nil {
				return "", fmt.Errorf("invalid place.number text: %v", err.Error())
			}
		} else {
			carPlaceNumbers, err = tanker.TemplateToString("place.numbers", p.texts.GetSingular("place.numbers", "ru"), params)
			if err != nil {
				return "", fmt.Errorf("invalid place.numbers text: %v", err.Error())
			}
		}
	}
	trainInfoParts := []string{}
	if carInfo != "" {
		trainInfoParts = append(trainInfoParts, carInfo)
	}
	if carType != "" {
		trainInfoParts = append(trainInfoParts, carType)
	}
	if carPlaceNumbers != "" {
		trainInfoParts = append(trainInfoParts, carPlaceNumbers)
	}
	return strings.Join(trainInfoParts, constants.MiddleDotSeparator), nil
}

func (p *Provider) getStationAndTimeZone(stationID string) (*rasp.TStation, *time.Location, error) {
	numericStationID, err := strconv.Atoi(stationID)
	if err != nil {
		return nil, nil, fmt.Errorf("unable to parse station id: %s", stationID)
	}
	station, found := p.stationDataProvider.GetStationByID(numericStationID)
	if !found {
		return nil, nil, fmt.Errorf("unknown station id: %s", stationID)
	}
	tz, found := p.timeZoneDataProvider.Get(int(station.TimeZoneId))
	if tz == nil || !found {
		return nil, nil, fmt.Errorf("unknown timezone for station: %s", stationID)
	}
	return station, tz, nil
}

func (p *Provider) getTrainStationInfo(station *rasp.TStation, stationTime, departureTime time.Time) (ui.TrainStationInfo, error) {
	trainStationInfo := ui.TrainStationInfo{}

	trainStationInfo.Station = getTitle(station)
	trainStationInfo.Time = getTimeString(stationTime)
	trainStationInfo.Date = getDateStringWithWeekday(stationTime)
	nextDayHint, err := p.getNextDateHint(stationTime, departureTime)
	if err != nil {
		return trainStationInfo, err
	}
	trainStationInfo.NextDateHint = nextDayHint

	return trainStationInfo, nil
}

func (p *Provider) getSettlementTitle(station *rasp.TStation) (int32, string) {
	if station == nil {
		return 0, ""
	}
	settlement, found := p.stationDataProvider.GetSettlementByStationID(int(station.Id))
	if !found || settlement == nil || settlement.TitleDefault == "" {
		return 0, getTitle(station)
	}
	return settlement.Id, settlement.TitleDefault
}

func getTrainName(trainOrderInfo travelapi.TrainOrderInfo) string {
	trainNameParts := []string{trainOrderInfo.TrainNumber}
	if trainOrderInfo.BrandTitle != "" {
		trainNameParts = append(trainNameParts, trainOrderInfo.BrandTitle)
	} else {
		trainNameParts = append(trainNameParts, trainOrderInfo.TrainStartSettlementTitle, "-", trainOrderInfo.TrainEndSettlementTitle)
	}
	return strings.Join(trainNameParts, " ")
}

func getTitle(station *rasp.TStation) string {
	var popularTitleRu, titleRu string
	if station.PopularTitle != nil {
		popularTitleRu = station.PopularTitle.Ru
	}
	if station.Title != nil {
		titleRu = station.Title.Ru
	}
	return strutil.Coalesce(
		station.PopularTitleDefault,
		popularTitleRu,
		station.TitleDefault,
		titleRu,
	)
}

func getTimeString(t time.Time) string {
	return t.Format("15:04")
}

func getDateStringWithWeekday(t time.Time) string {
	return fmt.Sprintf("%d&nbsp;%s, %s", t.Day(), constants.GenitiveMonthNames[t.Month()], constants.WeekdayFullNames[t.Weekday()])
}

func getDateString(t time.Time) string {
	return fmt.Sprintf("%d&nbsp;%s", t.Day(), constants.GenitiveMonthNames[t.Month()])
}

func (p *Provider) getNextDateHint(stationTime, departureTime time.Time) (string, error) {
	if !stationTime.After(departureTime) {
		return "", nil
	}
	depTime := getLocalMidday(departureTime)
	arrTime := getLocalMidday(stationTime)
	numDays := int(math.Round(arrTime.Sub(depTime).Hours() / 24.))
	switch {
	case numDays <= 0:
		return "", nil
	case numDays == 1:
		textParams := map[string]interface{}{
			"dateString": getDateString(stationTime),
		}
		return tanker.TemplateToString("next.day", p.texts.GetSingular("next.day", "ru"), textParams)
	default:
		textParams := map[string]interface{}{
			"daysShift":  strconv.Itoa(numDays),
			"daysPlural": p.texts.GetPlural("days.plural", "ru", numDays),
			"dateString": getDateString(stationTime),
		}
		return tanker.TemplateToString("next.days", p.texts.GetSingular("next.days", "ru"), textParams)
	}
}

func (p *Provider) getBlockPoints(departureStation, arrivalStation *rasp.TStation) ([]string, error) {
	departureSettlementID, departureSettlementTitle := p.getSettlementTitle(departureStation)
	if departureSettlementTitle == "" {
		return nil, fmt.Errorf("empty title for departure settlement %v", departureStation.Id)
	}
	arrivalSettlementID, arrivalSettlementTitle := p.getSettlementTitle(arrivalStation)
	if arrivalSettlementTitle == "" {
		return nil, fmt.Errorf("empty title for arrival settlement %v", arrivalStation.Id)
	}
	if departureSettlementID != arrivalSettlementID {
		return []string{departureSettlementTitle, arrivalSettlementTitle}, nil
	}
	return []string{getTitle(departureStation), getTitle(arrivalStation)}, nil
}

func getTime(strTime string, tz *time.Location) (time.Time, error) {
	t, err := time.Parse(time.RFC3339, strTime)
	return t.In(tz), err
}

func getLocalMidday(t time.Time) time.Time {
	return time.Date(t.Year(), t.Month(), t.Day(), 12, 0, 0, 0, time.UTC)
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.TrainOrderBlock
}
